import os
import flet as ft
import pygame
from mutagen.mp3 import MP3
import random
from mutagen.id3 import ID3, APIC
import json
import time
from threading import Timer
from functools import wraps
import logging
from datetime import datetime
import base64

# تنظیمات logging
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
    handlers=[
        logging.FileHandler('music_player.log', encoding='utf-8'),
        logging.StreamHandler()
    ]
)
logger = logging.getLogger(__name__)

SETTINGS_FILE = "player_state.json"
PLAYLISTS_FILE = "playlists.json"
CACHE_DIR = "../cache"

# ایجاد دایرکتوری کش
if not os.path.exists(CACHE_DIR):
    os.makedirs(CACHE_DIR)


def debounce(wait_time):
    """
    Decorator that will postpone a function's execution until after wait_time seconds
    have elapsed since the last time it was invoked.
    """

    def decorator(function):
        last_called = [0.0]

        @wraps(function)
        def debounced(*args, **kwargs):
            current_time = time.time()
            if current_time - last_called[0] >= wait_time:
                last_called[0] = current_time
                return function(*args, **kwargs)

        return debounced

    return decorator


def singleton(cls):
    """Decorator for singleton pattern"""
    instances = {}

    def get_instance(*args, **kwargs):
        if cls not in instances:
            instances[cls] = cls(*args, **kwargs)
        return instances[cls]

    return get_instance


@singleton
class CacheManager:
    """مدیریت کش برای آلبوم آرت و داده‌های دیگر"""

    def __init__(self):
        self.cache_dir = CACHE_DIR
        self.album_art_cache = {}

    def get_cache_key(self, file_path):
        """ایجاد کلید کش از مسیر فایل"""
        return base64.b64encode(file_path.encode()).decode()

    def save_album_art(self, file_path, image_data):
        """ذخیره آلبوم آرت در کش"""
        try:
            cache_key = self.get_cache_key(file_path)
            cache_file = os.path.join(self.cache_dir, f"{cache_key}.jpg")

            with open(cache_file, 'wb') as f:
                f.write(image_data)

            self.album_art_cache[cache_key] = image_data
            logger.info(f"Album art cached for {os.path.basename(file_path)}")
            return True
        except Exception as e:
            logger.error(f"Error caching album art: {e}")
            return False

    def load_album_art(self, file_path):
        """بارگذاری آلبوم آرت از کش"""
        try:
            cache_key = self.get_cache_key(file_path)

            # اول از مموری کش بررسی کن
            if cache_key in self.album_art_cache:
                return self.album_art_cache[cache_key]

            # سپس از فایل کش بررسی کن
            cache_file = os.path.join(self.cache_dir, f"{cache_key}.jpg")
            if os.path.exists(cache_file):
                with open(cache_file, 'rb') as f:
                    image_data = f.read()
                self.album_art_cache[cache_key] = image_data
                return image_data

            return None
        except Exception as e:
            logger.error(f"Error loading cached album art: {e}")
            return None

    def clear_cache(self):
        """پاک‌سازی کش"""
        try:
            for file in os.listdir(self.cache_dir):
                if file.endswith('.jpg'):
                    os.remove(os.path.join(self.cache_dir, file))
            self.album_art_cache.clear()
            logger.info("Cache cleared successfully")
            return True
        except Exception as e:
            logger.error(f"Error clearing cache: {e}")
            return False


class GlobalHotkeys:
    """مدیریت global hotkeys"""

    def __init__(self, player):
        self.player = player
        self.enabled = False
        self.key_press_time = {}

    def enable(self):
        """فعال‌سازی hotkeys"""
        self.enabled = True
        logger.info("Global hotkeys enabled")

    def handle_keyboard_event(self, e):
        """مدیریت رویدادهای کیبورد"""
        if not self.enabled:
            return

        key = e.key.lower() if e.key else ""
        current_time = time.time()

        logger.info(f"Processing key: '{e.key}'")

        # دکمه Play/Pause - فقط Media Play Pause اصلی
        if key in ['', 'media play pause', 'media_play_pause', 'play/pause']:
            logger.info("Play/Pause key detected")
            self.player.toggle_playback(None)

        # دکمه Next - °
        elif key in ['medianext', 'media_next', 'next_track', 'next', '°']:
            logger.info("Next key detected")
            self.player.play_next(None)

        # دکمه Previous - ±
        elif key in ['mediaprevious', 'media_previous', 'previous_track', 'previous', '±']:
            logger.info("Previous key detected")
            self.player.play_previous(None)

        # دکمه Stop
        elif key in ['mediastop', 'media_stop', 'stop']:
            logger.info("Stop key detected")
            self.player.stop_playback(None)




class MusicPlayerApp:
    def __init__(self, page: ft.Page):
        self.page = page
        self.is_playing = False
        self.repeat_mode = False
        self.shuffle_mode = False
        self.state_loaded = False
        self.loaded_from_state = False
        self.playlist = []
        self.current_song_index = 0
        self.duration = 0
        self.current_position = 0
        self.running = True
        self.current_playlist_name = "Default"
        self.favorites_playlist = "Favorites"
        self.playlists = {}
        self.favorite_songs = set()
        self.sidebar_visible = False
        self.progress_update_thread = None
        self.stop_progress_thread = False
        self._paused = False
        self.mini_player_mode = False
        self.search_query = ""

        # مدیر کش
        self.cache_manager = CacheManager()

        # hotkeys
        self.hotkeys = GlobalHotkeys(self)

        # تم‌ها
        self.themes = {
            "Blue Night": [ft.Colors.BLUE_900, ft.Colors.PURPLE_900],
            "Emerald Glow": [ft.Colors.GREEN_900, ft.Colors.TEAL_900],
            "Crimson Fire": [ft.Colors.RED_900, ft.Colors.ORANGE_900],
            "Sunset Orange": [ft.Colors.ORANGE_700, ft.Colors.PINK_900],
            "Midnight Purple": [ft.Colors.PURPLE_800, ft.Colors.INDIGO_900],
            "Dark Nebula": [ft.Colors.BLACK, ft.Colors.GREY_900],
            "Ocean Blue": [ft.Colors.CYAN_900, ft.Colors.BLUE_800],
            "Forest Green": [ft.Colors.GREEN_800, ft.Colors.LIGHT_GREEN_900]
        }
        self.current_theme = "Blue Night"

        # Initialize components
        self.initialize_page_settings()
        self.initialize_pygame()
        logger.info("App initialized successfully")
        self.build_ui()
        logger.info("UI built successfully")

        self.load_state()
        self.load_playlists()
        logger.info("State loaded successfully")

        # Event handlers
        self.page.on_resize = self.on_page_resize
        self.page.on_close = self.on_window_close
        self.page.window_prevent_close = True
        self.page.window.on_resized = self.on_window_resize
        self.page.window_prevent_close = True
        self.page.on_window_event = self.handle_window_event
        self.page.on_keyboard_event = self.on_keyboard

        # فعال‌سازی hotkeys
        self.hotkeys.enable()

        # شروع thread برای به‌روزرسانی پیشرفت
        self.start_progress_updater()

    def on_keyboard(self, e: ft.KeyboardEvent):
        """مدیریت رویدادهای کیبورد"""
        # لاگ کردن برای دیباگ
        debug_info = f"Key: {e.key}, Shift: {e.shift}, Ctrl: {e.ctrl}, Alt: {e.alt}, Meta: {e.meta}"
        logger.info(debug_info)
        print(debug_info)  # برای مشاهده در کنسول

        self.hotkeys.handle_keyboard_event(e)

    def handle_window_event(self, e):
        """مدیریت رویدادهای پنجره"""
        if e.data == "close":
            self.dispose()

    def start_progress_updater(self):
        """شروع thread برای به‌روزرسانی خودکار پیشرفت"""
        self.stop_progress_thread = False
        self.seek_position = None
        self.last_update_time = time.time()
        self.schedule_progress_update()

    def schedule_progress_update(self):
        """زمان‌بندی به‌روزرسانی بعدی"""
        if self.stop_progress_thread:
            return

        self.progress_timer = Timer(0.1, self.update_progress)
        self.progress_timer.daemon = True
        self.progress_timer.start()

    def update_progress(self):
        """به‌روزرسانی پیشرفت"""
        if self.stop_progress_thread:
            return

        try:
            if self.is_playing and self.duration > 0:
                current_time = time.time()
                time_diff = current_time - self.last_update_time

                if self.seek_position is not None:
                    self.current_position = self.seek_position + time_diff
                    self.seek_position = None
                else:
                    self.current_position += time_diff

                self.last_update_time = current_time

                if self.current_position > self.duration:
                    self.current_position = self.duration
                    self.handle_song_end()

                self.update_progress_ui()

        except Exception as e:
            logger.error(f"Progress update error: {e}")

        self.schedule_progress_update()

    @debounce(0.1)
    def update_progress_ui(self):
        """به‌روزرسانی UI با debounce"""
        try:
            # آپدیت progress bar اصلی
            if (hasattr(self, 'progress_slider') and
                    hasattr(self, 'progress_display') and
                    hasattr(self, 'current_time') and
                    hasattr(self, 'total_time')):
                progress_percent = self.current_position / self.duration if self.duration > 0 else 0

                self.progress_display.value = progress_percent
                self.progress_slider.value = self.current_position
                self.current_time.value = self.format_time(self.current_position)
                self.total_time.value = self.format_time(self.duration)

            # آپدیت mini-player progress
            if hasattr(self, 'progress_mini'):
                self.progress_mini.value = progress_percent

            # آپدیت زمان در mini-player
            if hasattr(self, 'mini_time_display'):
                self.mini_time_display.value = self.format_time(self.current_position)

            if self.page and self.page.controls:
                try:
                    self.progress_display.update()
                    self.progress_slider.update()
                    self.current_time.update()
                    self.total_time.update()

                    if hasattr(self, 'progress_mini'):
                        self.progress_mini.update()
                    if hasattr(self, 'mini_time_display'):
                        self.mini_time_display.update()

                except Exception as e:
                    logger.debug(f"UI update skipped: {e}")

        except Exception as e:
            logger.error(f"UI update error: {e}")


    def on_window_close(self, e):
        """Handle window close event"""
        self.dispose()

    def initialize_pygame(self):
        """Initialize Pygame mixer for audio playback"""
        pygame.mixer.init()
        pygame.mixer.music.set_volume(0.5)

    def build_ui(self):
        """Build the responsive user interface"""
        # بخش جستجو
        self.search_field = ft.TextField(
            hint_text="Search songs...",
            prefix_icon=ft.Icons.SEARCH,
            on_change=self.on_search_change,
            expand=True,
            text_size=14,
            border_color=ft.Colors.BLUE_400,
            focused_border_color=ft.Colors.BLUE_700,
        )

        self.header = self.create_header()
        self.album_cover = self.create_album_cover()
        self.song_title = self.create_song_title()
        self.progress_bar = self.create_progress_bar()
        self.playback_controls = self.create_playback_controls()
        self.volume_controls = self.create_volume_controls()

        # ایجاد song_list_header و playlist_controls
        self.song_list_header_text = ft.Text(
            value=self.current_playlist_name or "Playlist",
            size=18,
            weight=ft.FontWeight.BOLD,
            color=ft.Colors.WHITE,
            font_family="Montserrat"
        )

        self.song_list_header = ft.Row(
            [
                self.song_list_header_text,
                ft.Container(expand=True),
                ft.IconButton(
                    icon=ft.Icons.FILTER_LIST,
                    icon_color=ft.Colors.WHITE70,
                    tooltip="Sort songs",
                    on_click=self.open_sort_dialog,
                )
            ],
            alignment=ft.MainAxisAlignment.SPACE_BETWEEN,
        )

        self.playlist_controls = self.create_playlist_controls()
        self.song_list = self.create_song_list()

        # Main content column - با پشتیبانی از ارتفاع
        self.main_content = ft.Container(
            content=ft.ListView(  # تغییر از Column به ListView
                controls=[
                    self.header,
                    ft.Container(height=10),
                    self.album_cover,
                    ft.Container(height=10),
                    self.song_title,
                    ft.Container(height=5),
                    self.progress_bar,
                    ft.Container(height=10),
                    self.playback_controls,
                    ft.Container(height=10),
                    self.volume_controls,
                ],
                spacing=0,
                auto_scroll=False,
                expand=True,
            ),
            padding=15,
            expand=True,
        )

        # Song list container با پشتیبانی از ارتفاع
        self.song_list_container = ft.Container(
            content=ft.Column([
                ft.Row([
                    self.song_list_header,
                    ft.Container(width=10),
                    self.playlist_controls,
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                ft.Container(height=5),
                # ردیف جستجو
                ft.Row([
                    self.search_field,
                    ft.IconButton(
                        icon=ft.Icons.CLEAR,
                        icon_color=ft.Colors.WHITE70,
                        tooltip="Clear search",
                        on_click=self.clear_search,
                    )
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                ft.Container(height=5),
                ft.Container(
                    content=self.song_list,
                    expand=True,
                    border_radius=10,
                    padding=5,
                )
            ], spacing=10, expand=True),
            bgcolor=ft.Colors.with_opacity(0.08, ft.Colors.WHITE),
            border_radius=10,
            padding=10,
            width=350,
            height=400,  # ارتفاع اولیه
            visible=True,
        )

        # Song list container با قابلیت جستجو
        self.song_list_container = ft.Container(
            content=ft.Column([
                ft.Row([
                    self.song_list_header,
                    ft.Container(width=10),
                    self.playlist_controls,
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                ft.Container(height=5),
                # ردیف جستجو
                ft.Row([
                    self.search_field,
                    ft.IconButton(
                        icon=ft.Icons.CLEAR,
                        icon_color=ft.Colors.WHITE70,
                        tooltip="Clear search",
                        on_click=self.clear_search,
                    )
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                ft.Container(height=5),
                ft.Container(
                    content=self.song_list,
                    expand=True,
                    border_radius=10,
                    padding=5,
                )
            ], spacing=10, expand=True),
            bgcolor=ft.Colors.with_opacity(0.08, ft.Colors.WHITE),
            border_radius=10,
            padding=10,
            width=350,
            visible=True,
        )

        # Main layout با mini-player support
        self.main_layout = ft.Row(
            controls=[
                ft.Container(
                    content=self.main_content,
                    expand=True,
                ),
                ft.VerticalDivider(width=1, color=ft.Colors.with_opacity(0.3, ft.Colors.WHITE)),
                self.song_list_container,
            ],
            spacing=0,
            expand=True,
        )

        # Page container
        self.page_container = ft.Container(
            content=self.main_layout,
            gradient=ft.LinearGradient(
                begin=ft.alignment.top_left,
                end=ft.alignment.bottom_right,
                colors=self.themes[self.current_theme],
            ),
            expand=True,
            padding=10,
        )

        # Mini-player container (مخفی در ابتدا)
        self.mini_player_container = ft.Container(
            content=self.create_mini_player(),
            bgcolor=ft.Colors.BLUE_900,
            padding=10,
            visible=False,
            border_radius=10,
            shadow=ft.BoxShadow(blur_radius=10, color=ft.Colors.BLACK54)
        )

        # Layout اصلی
        self.page.add(
            ft.Column([
                self.page_container,
                self.mini_player_container
            ], expand=True)
        )

    def create_mini_player(self):
        """ایجاد mini-player پیشرفته"""
        # المان‌های mini-player
        self.album_image_mini = ft.Image(
            src="https://cdn-icons-png.flaticon.com/512/727/727240.png",
            width=45,
            height=45,
            border_radius=8,
            fit=ft.ImageFit.COVER,
        )

        self.song_label_mini = ft.Text(
            "No song playing",
            size=12,
            color=ft.Colors.WHITE,
            weight=ft.FontWeight.W_500,
            max_lines=1,
            overflow=ft.TextOverflow.ELLIPSIS,
            width=150
        )

        self.artist_label_mini = ft.Text(
            "Music Player",
            size=10,
            color=ft.Colors.WHITE70,
            max_lines=1,
            overflow=ft.TextOverflow.ELLIPSIS,
            width=150
        )

        # دکمه پلی/پاز در mini-player
        self.play_pause_btn_mini = ft.IconButton(
            icon=ft.Icons.PLAY_ARROW,
            icon_size=20,
            icon_color=ft.Colors.WHITE,
            on_click=self.toggle_playback,
            tooltip="Play/Pause"
        )

        # Progress bar برای mini-player
        self.progress_mini = ft.ProgressBar(
            value=0,
            width=200,
            height=3,
            color=ft.Colors.BLUE_400,
            bgcolor=ft.Colors.with_opacity(0.3, ft.Colors.WHITE),
        )

        self.favorite_btn_mini = ft.IconButton(
            icon=ft.Icons.FAVORITE_BORDER,
            icon_size=16,
            icon_color=ft.Colors.WHITE70,
            on_click=self.toggle_favorite_mini,
            tooltip="Add to favorites"
        )

        # ایجاد mini-player با ارتفاع دقیق
        mini_content = ft.Column([
            # ردیف اول: اطلاعات آهنگ و کنترل‌ها
            ft.Container(
                content=ft.Row([
                    ft.IconButton(
                        icon=ft.Icons.EXPAND,
                        icon_size=16,
                        icon_color=ft.Colors.WHITE,
                        on_click=self.toggle_mini_player,
                        tooltip="Expand to full player"
                    ),
                    self.album_image_mini,
                    ft.Column([
                        self.song_label_mini,
                        self.artist_label_mini,
                    ], spacing=0, expand=True),
                    self.favorite_btn_mini,
                    ft.IconButton(
                        icon=ft.Icons.CLOSE,
                        icon_size=16,
                        icon_color=ft.Colors.WHITE70,
                        on_click=self.close_mini_player,
                        tooltip="Close player"
                    ),
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                height=50
            ),

            # ردیف دوم: کنترل‌های پخش و progress
            ft.Container(
                content=ft.Row([
                    ft.IconButton(
                        icon=ft.Icons.SKIP_PREVIOUS,
                        icon_size=18,
                        icon_color=ft.Colors.WHITE,
                        on_click=self.play_previous,
                        tooltip="Previous song"
                    ),
                    self.play_pause_btn_mini,
                    ft.IconButton(
                        icon=ft.Icons.SKIP_NEXT,
                        icon_size=18,
                        icon_color=ft.Colors.WHITE,
                        on_click=self.play_next,
                        tooltip="Next song"
                    ),
                    ft.Container(
                        content=self.progress_mini,
                        expand=True,
                        margin=ft.margin.only(left=10, right=10)
                    ),
                    ft.Text(
                        "0:00",
                        size=10,
                        color=ft.Colors.WHITE70,
                        width=35
                    ),
                    ft.IconButton(
                        icon=ft.Icons.VOLUME_UP,
                        icon_size=16,
                        icon_color=ft.Colors.WHITE,
                        on_click=self.show_volume_popup_mini,
                        tooltip="Volume control"
                    ),
                ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),
                height=40
            ),
        ], spacing=0)  # spacing=0 برای حذف فاصله بین ردیف‌ها

        return ft.Container(
            content=mini_content,
            bgcolor=ft.Colors.BLUE_900,
            padding=10,
            border_radius=10,
            shadow=ft.BoxShadow(blur_radius=15, color=ft.Colors.BLACK54),
            width=400,
            height=100  # ارتفاع ثابت برای mini-player
        )

    def toggle_mini_player(self, e=None):
        """تغییر حالت mini-player با انیمیشن"""
        self.mini_player_mode = not self.mini_player_mode

        if self.mini_player_mode:
            # رفتن به حالت mini-player
            self.page_container.visible = False
            self.mini_player_container.visible = True

            # تنظیم پنجره به اندازه دقیق mini-player
            self.page.window.width = 420
            self.page.window.height = 100  # ارتفاع ثابت
            self.page.window.resizable = False

            # تنظیم mini-player برای پر کردن کل پنجره
            self.mini_player_container.width = 420
            self.mini_player_container.height = 100
            self.mini_player_container.expand = True

            # همگام‌سازی وضعیت با پخش کننده اصلی
            self.sync_mini_player_state()

            self.show_message("Mini player activated")
            logger.info("Switched to mini player mode")

        else:
            # بازگشت به حالت عادی
            self.page_container.visible = True
            self.mini_player_container.visible = False
            self.page.window.width = 1000
            self.page.window.height = 700
            self.page.window.resizable = True

            self.show_message("Full player activated")
            logger.info("Switched to full player mode")

        self.safe_page_update()

    def sync_mini_player_state(self):
        """همگام‌سازی وضعیت mini-player با پخش کننده اصلی"""
        if not self.playlist:
            return

        # آپدیت دکمه پلی/پاز
        self.play_pause_btn_mini.icon = ft.Icons.PAUSE if self.is_playing else ft.Icons.PLAY_ARROW

        # آپدیت علاقه‌مندی
        current_song = self.playlist[self.current_song_index] if self.playlist else None
        if current_song and hasattr(self, 'favorite_btn_mini'):
            is_favorite = current_song in self.favorite_songs
            self.favorite_btn_mini.icon = ft.Icons.FAVORITE if is_favorite else ft.Icons.FAVORITE_BORDER
            self.favorite_btn_mini.icon_color = ft.Colors.RED_400 if is_favorite else ft.Colors.WHITE70

        # آپدیت اطلاعات آهنگ
        if current_song and hasattr(self, 'song_label_mini'):
            song_name = self.extract_song_metadata(current_song)
            self.song_label_mini.value = song_name

    def toggle_favorite_mini(self, e):
        """تغییر علاقه‌مندی از mini-player"""
        if self.playlist and self.current_song_index < len(self.playlist):
            current_song = self.playlist[self.current_song_index]
            self.toggle_favorite(current_song)
            self.sync_mini_player_state()

    def close_mini_player(self, e):
        """بستن mini-player و خروج از برنامه"""

        def confirm_exit(e):
            self.dispose()
            self.page.window_destroy()

        dlg = ft.AlertDialog(
            title=ft.Text("Exit Player", color=ft.Colors.WHITE),
            content=ft.Text("Are you sure you want to close the music player?", color=ft.Colors.WHITE70),
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg)),
                ft.TextButton("Exit", on_click=confirm_exit, style=ft.ButtonStyle(color=ft.Colors.RED_400)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def show_volume_popup_mini(self, e):
        """نمایش پاپ‌آپ کنترل صدا برای mini-player"""
        volume_slider = ft.Slider(
            min=0,
            max=1,
            value=self.volume_slider.value,
            divisions=10,
            on_change=self.adjust_volume,
            width=150,
        )

        dlg = ft.AlertDialog(
            content=ft.Column([
                ft.Text("Volume Control", size=14, color=ft.Colors.WHITE, weight=ft.FontWeight.BOLD),
                volume_slider
            ], tight=True, horizontal_alignment=ft.CrossAxisAlignment.CENTER),
            actions=[
                ft.TextButton("Close", on_click=lambda e: self.page.close(dlg))
            ],
            bgcolor=ft.Colors.BLUE_800,
        )
        self.page.open(dlg)

    def show_volume_popup(self, e):
        """نمایش پاپ‌آپ کنترل صدا در mini-player"""
        volume_slider = ft.Slider(
            min=0,
            max=1,
            value=self.volume_slider.value,
            divisions=10,
            on_change=self.adjust_volume,
            width=200,
        )

        dlg = ft.AlertDialog(
            content=ft.Column([
                ft.Text("Volume", color=ft.Colors.WHITE),
                volume_slider
            ], tight=True),
            actions=[
                ft.TextButton("Close", on_click=lambda e: self.page.close(dlg))
            ]
        )
        self.page.open(dlg)

    def create_header(self):
        """Create a modern header with additional features"""
        self.file_picker = ft.FilePicker(on_result=self.handle_directory_pick)
        self.backup_picker = ft.FilePicker(on_result=self.handle_backup_save)
        self.restore_picker = ft.FilePicker(on_result=self.handle_backup_restore)
        self.page.overlay.extend([self.file_picker, self.backup_picker, self.restore_picker])

        return ft.Container(
            content=ft.Row(
                controls=[
                    ft.Row([
                        ft.IconButton(
                            ft.Icons.FOLDER_OPEN,
                            icon_color=ft.Colors.WHITE,
                            bgcolor=ft.Colors.BLUE_700,
                            tooltip="Open Folder",
                            on_click=lambda _: self.file_picker.get_directory_path(),
                            style=ft.ButtonStyle(shape=ft.CircleBorder(), padding=8),
                        ),
                        ft.IconButton(
                            ft.Icons.COMPRESS,
                            icon_color=ft.Colors.WHITE,
                            bgcolor=ft.Colors.GREEN_700,
                            tooltip="Mini Player",
                            on_click=self.toggle_mini_player,
                            style=ft.ButtonStyle(shape=ft.CircleBorder(), padding=8),
                        ),
                    ]),

                    ft.Text("",
                            size=20,
                            weight=ft.FontWeight.BOLD,
                            color=ft.Colors.WHITE,
                            font_family="Montserrat"),

                    ft.Row([
                        ft.IconButton(
                            ft.Icons.INFO_OUTLINE,
                            icon_color=ft.Colors.WHITE,
                            tooltip="App Info",
                            on_click=self.show_app_info,
                        ),
                        ft.IconButton(
                            ft.Icons.SETTINGS,
                            icon_color=ft.Colors.WHITE,
                            bgcolor=ft.Colors.BLUE_700,
                            tooltip="Settings",
                            on_click=self.open_settings_popup,
                            style=ft.ButtonStyle(shape=ft.CircleBorder(), padding=8),
                        ),
                    ]),
                ],
                alignment=ft.MainAxisAlignment.SPACE_BETWEEN,
                spacing=20,
            ),
            padding=10,
        )

    def show_app_info(self, e):
        """نمایش اطلاعات برنامه"""
        info_text = """Advanced Music Player

    Features:
    • Play MP3 files with metadata support
    • Multiple playlists management
    • Favorites system
    • Theme customization
    • Mini-player mode
    • Global hotkeys support
    • Search and sort functionality
    • Backup and restore
    • Album art display
    • Shuffle and repeat modes

    Shortcuts:
    Media Play/Pause - Play/Pause
    Media Next - Next song
    Media Previous - Previous song
    Media Stop - Stop playback
    """
        # بقیه کد...
        dlg = ft.AlertDialog(
            title=ft.Text("App Information", color=ft.Colors.WHITE),
            content=ft.Container(
                content=ft.Text(info_text, color=ft.Colors.WHITE70, size=12),
                width=400,
                padding=10,
            ),
            actions=[
                ft.TextButton("Close", on_click=lambda e: self.page.close(dlg))
            ]
        )
        self.page.open(dlg)

    def open_settings_popup(self, e):
        """Open enhanced settings popup"""
        theme_dropdown = ft.Dropdown(
            label="Select Theme",
            value=self.current_theme,
            options=[ft.dropdown.Option(theme) for theme in self.themes.keys()],
            on_change=self.change_theme,
            width=200,
            color=ft.Colors.WHITE,
            border_color=ft.Colors.BLUE_400,
        )

        dlg = ft.AlertDialog(
            title=ft.Text("Settings", color=ft.Colors.WHITE),
            content=ft.Container(
                content=ft.Column([
                    ft.Text("Theme", color=ft.Colors.WHITE, weight=ft.FontWeight.BOLD),
                    theme_dropdown,

                    ft.Divider(color=ft.Colors.WHITE24),


                    ft.Text("Backup & Restore", color=ft.Colors.WHITE, weight=ft.FontWeight.BOLD),
                    ft.ElevatedButton("Backup Playlists",
                                      on_click=lambda _: self.backup_picker.save_file(
                                          file_name="playlists_backup.json")),
                    ft.ElevatedButton("Restore Playlists",
                                      on_click=lambda _: self.restore_picker.pick_files(
                                          allow_multiple=False,
                                          file_type=ft.FilePickerFileType.CUSTOM,
                                          allowed_extensions=["json"])),

                    ft.Divider(color=ft.Colors.WHITE24),

                    ft.Text("Cache Management", color=ft.Colors.WHITE, weight=ft.FontWeight.BOLD),
                    ft.ElevatedButton("Clear Cache", on_click=self.clear_cache),

                ], spacing=10),
                bgcolor=ft.Colors.BLUE_900,
                border_radius=10,
                padding=15,
                width=350,
            ),
            actions=[
                ft.TextButton("Close", on_click=lambda e: self.page.close(dlg),
                              style=ft.ButtonStyle(color=ft.Colors.WHITE)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def clear_cache(self, e):
        """پاک‌سازی کش"""
        if self.cache_manager.clear_cache():
            self.show_message("Cache cleared successfully!")
        else:
            self.show_message("Error clearing cache!")

    def change_theme(self, e):
        """Change the app theme"""
        self.current_theme = e.control.value
        self.page_container.gradient = ft.LinearGradient(
            begin=ft.alignment.top_left,
            end=ft.alignment.bottom_right,
            colors=self.themes[self.current_theme]
        )
        self.save_state()
        self.safe_page_update()

    def handle_backup_save(self, e: ft.FilePickerResultEvent):
        """Handle saving playlists backup"""
        if e.path:
            try:
                data = {
                    "playlists": self.playlists,
                    "favorites": list(self.favorite_songs),
                    "backup_date": datetime.now().isoformat(),
                    "version": "2.0"
                }
                with open(e.path, "w", encoding="utf-8") as f:
                    json.dump(data, f, ensure_ascii=False, indent=2)
                self.show_message(f"Backup saved to {e.path}")
                logger.info(f"Backup created: {e.path}")
            except Exception as ex:
                self.show_message(f"Backup error: {str(ex)}")
                logger.error(f"Backup error: {ex}")

    def handle_backup_restore(self, e: ft.FilePickerResultEvent):
        """Handle restoring playlists from backup"""
        if e.files and e.files[0].path:
            try:
                with open(e.files[0].path, "r", encoding="utf-8") as f:
                    data = json.load(f)

                self.playlists = data.get("playlists", {"Default": [], "Favorites": []})
                self.favorite_songs = set(data.get("favorites", []))
                self.current_playlist_name = "Default"
                self.playlist = self.playlists[self.current_playlist_name].copy()
                self.current_song_index = 0
                self.update_song_list()
                self.update_song_list_header()

                if self.playlist:
                    self.play_song(0)
                else:
                    self.song_label.value = "No songs in playlist"
                    self.album_image.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"
                    self.is_playing = False
                    self.play_pause_btn.icon = ft.Icons.PLAY_ARROW

                self.save_playlists()
                self.show_message("Playlists restored successfully!")
                logger.info("Playlists restored from backup")
                self.safe_page_update()
            except Exception as ex:
                self.show_message(f"Restore error: {str(ex)}")
                logger.error(f"Restore error: {ex}")

    def create_playlist_controls(self):
        """Create enhanced playlist controls"""
        return ft.Row([
            ft.IconButton(
                ft.Icons.PLAYLIST_PLAY,
                icon_color=ft.Colors.WHITE,
                tooltip="Select playlist",
                on_click=self.open_playlist_popup,
            ),
            ft.IconButton(
                ft.Icons.ADD,
                icon_color=ft.Colors.WHITE,
                tooltip="Create new playlist",
                on_click=self.open_new_playlist_popup,
            ),
            ft.IconButton(
                ft.Icons.DELETE,
                icon_color=ft.Colors.RED_400,
                tooltip="Delete current playlist",
                on_click=self.delete_current_playlist,
            ),
            ft.IconButton(
                ft.Icons.CREATE_NEW_FOLDER,
                icon_color=ft.Colors.WHITE,
                tooltip="Create playlist from current folder",
                on_click=self.open_new_playlist_from_folder_popup,
            ),
            ft.IconButton(
                ft.Icons.MENU,
                icon_color=ft.Colors.WHITE,
                tooltip="Toggle playlist",
                on_click=self.toggle_sidebar,
                visible=False,
            ),
        ], spacing=5, alignment=ft.MainAxisAlignment.END)

    def toggle_sidebar(self, e):
        """Toggle sidebar visibility"""
        self.sidebar_visible = not self.sidebar_visible
        self.song_list_container.visible = self.sidebar_visible
        self.update_layout()

    def open_playlist_popup(self, e):
        """Open playlist selection popup"""
        playlist_list = ft.ListView(
            expand=True,
            spacing=5,
            padding=10,
        )

        for name in self.playlists.keys():
            item_count = len(self.playlists[name])
            playlist_list.controls.append(
                ft.ListTile(
                    title=ft.Text(name, color=ft.Colors.WHITE),
                    subtitle=ft.Text(f"{item_count} songs", color=ft.Colors.WHITE70, size=12),
                    on_click=lambda e, n=name: self.select_playlist(n, e.control.parent.parent.parent),
                )
            )

        dlg = ft.AlertDialog(
            title=ft.Text("Select Playlist", color=ft.Colors.WHITE),
            content=ft.Container(
                content=playlist_list,
                height=300,
                bgcolor=ft.Colors.BLUE_900,
                border_radius=10,
            ),
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg),
                              style=ft.ButtonStyle(color=ft.Colors.WHITE)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def select_playlist(self, playlist_name, dlg):
        """Select a playlist"""
        if playlist_name in self.playlists:
            self.current_playlist_name = playlist_name
            self.playlist = self.playlists[playlist_name].copy()
            self.current_song_index = 0
            self.update_song_list()
            self.update_song_list_header()

            if self.playlist:
                self.play_song(0)
            else:
                self.song_label.value = "No songs in playlist"
                self.album_image.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"
                self.is_playing = False
                self.play_pause_btn.icon = ft.Icons.PLAY_ARROW
                self.safe_page_update()

            self.page.close(dlg)
            self.save_playlists()

    def open_new_playlist_popup(self, e):
        """Open new playlist creation popup"""
        self.new_playlist_field = ft.TextField(
            hint_text="Enter playlist name",
            width=200,
            color=ft.Colors.WHITE,
            border_color=ft.Colors.BLUE_400,
            text_size=12,
        )

        def confirm_create(e):
            playlist_name = self.new_playlist_field.value.strip()
            if playlist_name and playlist_name not in self.playlists:
                self.playlists[playlist_name] = []
                self.current_playlist_name = playlist_name
                self.playlist = self.playlists[playlist_name].copy()
                self.current_song_index = 0
                self.update_song_list()
                self.update_song_list_header()
                self.show_message(f"Playlist '{playlist_name}' created!")
                self.save_playlists()
                self.page.close(dlg)
            elif playlist_name in self.playlists:
                self.show_message("Playlist already exists!")
            else:
                self.show_message("Please enter a valid playlist name!")

        dlg = ft.AlertDialog(
            title=ft.Text("Create New Playlist", color=ft.Colors.WHITE),
            content=ft.Container(
                content=self.new_playlist_field,
                bgcolor=ft.Colors.BLUE_900,
                border_radius=10,
                padding=10,
            ),
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg),
                              style=ft.ButtonStyle(color=ft.Colors.WHITE)),
                ft.TextButton("Create", on_click=confirm_create, style=ft.ButtonStyle(color=ft.Colors.BLUE_400)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def open_new_playlist_from_folder_popup(self, e):
        """Open create playlist from folder popup"""
        self.new_playlist_from_folder_field = ft.TextField(
            hint_text="Enter playlist name",
            width=200,
            color=ft.Colors.WHITE,
            border_color=ft.Colors.BLUE_400,
            text_size=12,
        )

        def confirm_create_from_folder(e):
            playlist_name = self.new_playlist_from_folder_field.value.strip()
            if playlist_name and playlist_name not in self.playlists:
                self.playlists[playlist_name] = self.playlist.copy()
                self.current_playlist_name = playlist_name
                self.playlist = self.playlists[playlist_name].copy()
                self.current_song_index = 0
                self.update_song_list()
                self.update_song_list_header()
                self.show_message(f"Playlist '{playlist_name}' created with {len(self.playlist)} songs!")
                self.save_playlists()
                self.page.close(dlg)
            elif playlist_name in self.playlists:
                self.show_message("Playlist already exists!")
            else:
                self.show_message("Please enter a valid playlist name!")

        dlg = ft.AlertDialog(
            title=ft.Text("Create Playlist from Folder", color=ft.Colors.WHITE),
            content=ft.Container(
                content=self.new_playlist_from_folder_field,
                bgcolor=ft.Colors.BLUE_900,
                border_radius=10,
                padding=10,
            ),
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg),
                              style=ft.ButtonStyle(color=ft.Colors.WHITE)),
                ft.TextButton("Create", on_click=confirm_create_from_folder,
                              style=ft.ButtonStyle(color=ft.Colors.BLUE_400)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def create_album_cover(self):
        """Create album cover with enhanced styling"""
        self.album_image = ft.Image(
            src="https://cdn-icons-png.flaticon.com/512/727/727240.png",
            width=200,
            height=200,
            fit=ft.ImageFit.CONTAIN,
            border_radius=15,
        )
        return ft.Container(
            content=self.album_image,
            alignment=ft.alignment.center,
            border_radius=15,
            shadow=ft.BoxShadow(
                blur_radius=15,
                spread_radius=2,
                color=ft.Colors.with_opacity(0.4, ft.Colors.BLACK)
            ),
        )

    def create_song_title(self):
        """Create enhanced song title display"""
        self.song_label = ft.Text(
            "No song selected",
            size=15,
            color=ft.Colors.WHITE,
            weight=ft.FontWeight.BOLD,
            text_align=ft.TextAlign.CENTER,
            font_family="Montserrat",
        )

        self.artist_label = ft.Text(
            "Unknown Artist",
            size=12,
            color=ft.Colors.WHITE70,
            text_align=ft.TextAlign.CENTER,
        )

        return ft.Container(
            content=ft.Column([
                self.song_label,
                self.artist_label
            ], spacing=2, horizontal_alignment=ft.CrossAxisAlignment.CENTER),
            padding=6,
            alignment=ft.alignment.center,
        )

    def create_progress_bar(self):
        """Create enhanced progress bar"""
        self.current_time = ft.Text("0:00", color=ft.Colors.WHITE70, size=12)
        self.total_time = ft.Text("0:00", color=ft.Colors.WHITE70, size=12)

        self.progress_display = ft.ProgressBar(
            value=0,
            expand=True,
            height=6,
            color=ft.Colors.GREEN_400,
            bgcolor=ft.Colors.with_opacity(0.3, ft.Colors.WHITE),
        )

        self.progress_slider = ft.Slider(
            min=0,
            max=100,
            value=0,
            expand=True,
            active_color=ft.Colors.BLUE_400,
            inactive_color=ft.Colors.with_opacity(0.3, ft.Colors.WHITE),
            thumb_color=ft.Colors.WHITE,
            on_change=self.on_progress_change,
            on_change_end=self.seek_track,
        )

        return ft.Container(
            content=ft.Column(
                [
                    ft.Row([
                        self.current_time,
                        self.progress_display,
                        self.total_time
                    ],
                        alignment=ft.MainAxisAlignment.SPACE_BETWEEN,
                        vertical_alignment=ft.CrossAxisAlignment.CENTER,
                        spacing=5),
                    self.progress_slider,
                ],
                spacing=2,
            ),
            padding=2,
            alignment=ft.alignment.center,
        )

    def create_playback_controls(self):
        """Create enhanced playback controls"""
        self.play_pause_btn = ft.IconButton(
            ft.Icons.PLAY_ARROW,
            icon_size=32,
            icon_color=ft.Colors.WHITE,
            bgcolor=ft.Colors.BLUE_700,
            on_click=self.toggle_playback,
            style=ft.ButtonStyle(
                shape=ft.CircleBorder(),
                padding=10,
                overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
            ),
        )

        self.stop_btn = ft.IconButton(
            ft.Icons.STOP,
            icon_size=25,
            icon_color=ft.Colors.WHITE,
            on_click=self.stop_playback,
            style=ft.ButtonStyle(
                shape=ft.CircleBorder(),
                padding=8,
                overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
            ),
        )

        self.repeat_btn = ft.IconButton(
            ft.Icons.REPEAT,
            icon_size=20,
            icon_color=ft.Colors.WHITE70,
            on_click=self.toggle_repeat,
            style=ft.ButtonStyle(
                shape=ft.CircleBorder(),
                padding=8,
                overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
            ),
        )
        self.shuffle_btn = ft.IconButton(
            ft.Icons.SHUFFLE,
            icon_size=20,
            icon_color=ft.Colors.WHITE70,
            on_click=self.toggle_shuffle,
            style=ft.ButtonStyle(
                shape=ft.CircleBorder(),
                padding=8,
                overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
            ),
        )

        return ft.Row(
            controls=[
                self.shuffle_btn,
                ft.IconButton(
                    ft.Icons.SKIP_PREVIOUS,
                    icon_size=25,
                    icon_color=ft.Colors.WHITE,
                    on_click=self.play_previous,
                    style=ft.ButtonStyle(
                        shape=ft.CircleBorder(),
                        padding=8,
                        overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
                    ),
                ),
                self.stop_btn,
                self.play_pause_btn,
                ft.IconButton(
                    ft.Icons.SKIP_NEXT,
                    icon_size=25,
                    icon_color=ft.Colors.WHITE,
                    on_click=self.play_next,
                    style=ft.ButtonStyle(
                        shape=ft.CircleBorder(),
                        padding=8,
                        overlay_color=ft.Colors.with_opacity(0.1, ft.Colors.WHITE),
                    ),
                ),
                self.repeat_btn,
            ],
            alignment=ft.MainAxisAlignment.CENTER,
            spacing=10,
        )

    def create_volume_controls(self):
        """Create volume controls"""
        self.volume_slider = ft.Slider(
            min=0,
            max=1,
            value=0.5,
            divisions=10,
            on_change=self.adjust_volume,
            width=200,
            active_color=ft.Colors.BLUE_400,
            inactive_color=ft.Colors.with_opacity(0.3, ft.Colors.WHITE),
            thumb_color=ft.Colors.WHITE,
        )
        return ft.Row(
            controls=[
                ft.Icon(ft.Icons.VOLUME_UP, color=ft.Colors.WHITE70),
                self.volume_slider,
            ],
            alignment=ft.MainAxisAlignment.CENTER,
            spacing=10,
        )

    def update_song_list_header(self):
        """Update song list header"""
        if hasattr(self, 'song_list_header_text'):
            self.song_list_header_text.value = self.current_playlist_name or "Playlist"
            self.safe_page_update()

    def create_song_list(self):
        """Create the song list view"""
        self.song_list = ft.ListView(
            spacing=2,
            padding=5,
            expand=True,
            auto_scroll=False,
        )
        return self.song_list

    def on_search_change(self, e):
        """مدیریت تغییرات جستجو"""
        self.search_query = e.control.value.lower().strip()
        self.update_song_list()

    def clear_search(self, e):
        """پاک‌کردن جستجو"""
        self.search_field.value = ""
        self.search_query = ""
        self.update_song_list()
        self.safe_page_update()

    def open_sort_dialog(self, e):
        """نمایش دیالوگ مرتب‌سازی"""
        sort_options = ft.RadioGroup(
            content=ft.Column([
                ft.Radio(value="name", label="By Name"),
                ft.Radio(value="date", label="By Date Added"),
                ft.Radio(value="duration", label="By Duration"),
            ])
        )

        def apply_sort(e):
            self.sort_songs(sort_options.value)
            self.page.close(dlg)

        dlg = ft.AlertDialog(
            title=ft.Text("Sort Songs", color=ft.Colors.WHITE),
            content=sort_options,
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg)),
                ft.TextButton("Apply", on_click=apply_sort),
            ]
        )
        self.page.open(dlg)

    def sort_songs(self, sort_by):
        """مرتب‌سازی آهنگ‌ها"""
        if not self.playlist:
            return

        try:
            if sort_by == "name":
                self.playlist.sort(key=lambda x: os.path.basename(x).lower())
            elif sort_by == "date":
                # برای سادگی، از زمان modification استفاده می‌کنیم
                self.playlist.sort(key=lambda x: os.path.getmtime(x), reverse=True)
            elif sort_by == "duration":
                # مرتب‌سازی بر اساس مدت زمان آهنگ
                def get_duration(path):
                    try:
                        audio = MP3(path)
                        return audio.info.length
                    except:
                        return 0

                self.playlist.sort(key=get_duration)

            self.update_song_list()
            self.show_message(f"Songs sorted by {sort_by}")
            logger.info(f"Songs sorted by {sort_by}")
        except Exception as e:
            logger.error(f"Sort error: {e}")
            self.show_message("Error sorting songs")

    def scroll_to_current_song(self):
        """اسکرول به آهنگ جاری"""
        if not self.playlist or self.current_song_index >= len(self.playlist):
            return

        try:
            item_height = 52
            viewport_height = 400
            current_pos = self.current_song_index * item_height
            target_offset = current_pos - (viewport_height / 2) + (item_height / 2)
            target_offset = max(0, target_offset)

            self.song_list.scroll_to(
                offset=target_offset,
                duration=400,
                curve=ft.AnimationCurve.EASE_OUT
            )
        except Exception as e:
            logger.error(f"Smart scroll error: {e}")

    def update_layout(self):
        """Update layout based on window size for both width and height"""
        try:
            width = self.page.window_width
            height = self.page.window_height

            # تنظیم حداقل اندازه
            min_width = 400
            min_height = 300  # کاهش حداقل ارتفاع برای انعطاف‌پذیری بیشتر

            if width < min_width or height < min_height:
                self.page.window_width = max(width, min_width)
                self.page.window_height = max(height, min_height)
                return

            # مدیریت تغییرات عرض
            if width <= 768:
                self.main_layout.controls[1].visible = False
                self.main_layout.controls[0].expand = True
                self.playlist_controls.controls[-1].visible = True
                self.song_list_container.width = min(300, width - 20)
                self.song_list_container.alignment = ft.alignment.top_right
            else:
                self.main_layout.controls[1].visible = True
                self.main_layout.controls[0].expand = True
                self.main_layout.controls[1].width = min(350, width * 0.4)
                self.playlist_controls.controls[-1].visible = False
                self.sidebar_visible = False
                self.song_list_container.visible = True
                self.song_list_container.alignment = ft.alignment.center

            # مدیریت تغییرات ارتفاع
            album_size = 250  # اندازه پیش‌فرض
            show_volume_controls = True
            header_buttons_visible = True
            song_list_visible = True
            padding = 15
            spacing = 10

            if height <= 400:
                # حالت خیلی کوچک
                album_size = 80
                header_buttons_visible = False
                show_volume_controls = False
                song_list_visible = False
                padding = 5
                spacing = 2
                self.song_title.content.controls[0].size = 12
                self.song_title.content.controls[1].size = 10
                self.playback_controls.controls[0].icon_size = 20
                self.playback_controls.controls[1].icon_size = 20
                self.playback_controls.controls[2].icon_size = 20
                self.playback_controls.controls[3].icon_size = 20
                self.playback_controls.controls[4].icon_size = 20

            elif height <= 500:
                # حالت کوچک
                album_size = 120
                show_volume_controls = False
                song_list_visible = True
                padding = 8
                spacing = 5
                self.song_title.content.controls[0].size = 14
                self.song_title.content.controls[1].size = 12
                self.playback_controls.controls[0].icon_size = 25
                self.playback_controls.controls[1].icon_size = 25
                self.playback_controls.controls[2].icon_size = 25
                self.playback_controls.controls[3].icon_size = 25
                self.playback_controls.controls[4].icon_size = 25

            elif height <= 600:
                # حالت متوسط
                album_size = 180
                show_volume_controls = True
                song_list_visible = True
                padding = 10
                spacing = 8
                self.song_title.content.controls[0].size = 16
                self.song_title.content.controls[1].size = 14
                self.playback_controls.controls[0].icon_size = 30
                self.playback_controls.controls[1].icon_size = 30
                self.playback_controls.controls[2].icon_size = 30
                self.playback_controls.controls[3].icon_size = 30
                self.playback_controls.controls[4].icon_size = 30

            else:
                # حالت بزرگ
                album_size = 250
                padding = 15
                spacing = 10
                self.song_title.content.controls[0].size = 18
                self.song_title.content.controls[1].size = 16
                self.playback_controls.controls[0].icon_size = 40
                self.playback_controls.controls[1].icon_size = 40
                self.playback_controls.controls[2].icon_size = 40
                self.playback_controls.controls[3].icon_size = 40
                self.playback_controls.controls[4].icon_size = 40

            # اعمال تنظیمات
            self.album_cover.content.width = album_size
            self.album_cover.content.height = album_size
            self.main_content.padding = padding
            self.main_content.content.spacing = spacing
            self.header.content.controls[0].visible = header_buttons_visible
            self.header.content.controls[2].visible = header_buttons_visible
            self.volume_controls.visible = show_volume_controls
            self.song_list_container.visible = song_list_visible

            # تنظیم ارتفاع song_list_container
            self.song_list_container.height = min(height * 0.7, height - 150)

            # تنظیم ارتفاع song_list
            if hasattr(self, 'song_list'):
                max_song_list_height = self.song_list_container.height - 120
                self.song_list.height = min(max_song_list_height, height - 150)

            if self.playlist:
                self.page.run_task(self.delayed_scroll)

            self.safe_page_update()

        except Exception as e:
            logger.error(f"Layout update error: {e}")

    def initialize_page_settings(self):
        """Initialize page settings with modern styling and responsive support"""
        self.page.title = "Music Player"
        self.page.window.width = 850
        self.page.window.height = 620
        self.page.window.min_width = 400  # حداقل عرض
        self.page.window.min_height = 500  # حداقل ارتفاع
        self.page.window.resizable = True
        self.page.padding = 0
        self.page.bgcolor = ft.Colors.BLACK
        self.page.fonts = {
            "Roboto": "https://fonts.googleapis.com/css2?family=Roboto:wght@400;500;700&display=swap",
            "Montserrat": "https://fonts.googleapis.com/css2?family=Montserrat:wght@400;500;600;700&display=swap"
        }
        self.page.theme = ft.Theme(font_family="Roboto")
        self.page.theme_mode = ft.ThemeMode.DARK

        # فعال‌سازی responsive برای هر دو بعد
        self.page.on_resize = self.on_page_resize

    def on_page_resize(self, e):
        """Handle page resize for both dimensions"""
        self.update_layout()

    def on_window_resize(self, e):
        """Handle window resize for both dimensions"""
        self.update_layout()

    def delayed_scroll(self):
        """اسکرول با تاخیر"""
        import time
        time.sleep(0.05)
        self.scroll_to_current_song()

    def load_playlists(self):
        """Load playlists from file"""
        if not os.path.exists(PLAYLISTS_FILE):
            self.playlists = {
                "Default": [],
                "Favorites": []
            }
            return

        try:
            with open(PLAYLISTS_FILE, "r", encoding="utf-8") as f:
                data = json.load(f)

            self.playlists = data.get("playlists", {})
            self.favorite_songs = set(data.get("favorites", []))

            if "Default" not in self.playlists:
                self.playlists["Default"] = []
            if "Favorites" not in self.playlists:
                self.playlists["Favorites"] = []

            self.update_song_list()
        except Exception as e:
            logger.error(f"Load playlists error: {e}")
            self.playlists = {
                "Default": [],
                "Favorites": []
            }

    def load_songs(self, path):
        """Load songs from directory"""
        try:
            self.playlist = []
            supported_formats = (".mp3", ".wav", ".ogg", ".flac")  # فرمت‌های پشتیبانی شده

            for root, _, files in os.walk(path):
                for file in files:
                    if file.lower().endswith(supported_formats):
                        self.playlist.append(os.path.join(root, file))

            self.playlist.sort()
            self.current_song_index = 0
            self.update_song_list()

            self.loaded_from_state = False

            if self.playlist:
                self.play_song(0)
                self.save_state()
                self.show_message(f"Loaded {len(self.playlist)} songs!")
                self.add_songs_to_playlist("Default", self.playlist)
            else:
                self.show_message("No supported audio files found!")

        except Exception as e:
            logger.error(f"Load songs error: {e}")
            self.show_message(f"Error: {str(e)}")

    def play_song(self, index):
        """Play a song"""
        if not self.playlist:
            return

        try:
            if index < 0 or index >= len(self.playlist):
                index = 0

            self.current_song_index = index
            song = self.playlist[self.current_song_index]

            pygame.mixer.music.stop()
            pygame.mixer.music.load(song)
            pygame.mixer.music.play()
            self.is_playing = True
            self._paused = False

            self.current_position = 0
            self.seek_position = None
            self.last_update_time = time.time()

            try:
                audio = MP3(song)
                self.duration = audio.info.length
            except:
                self.duration = 0

            self.progress_slider.max = self.duration if self.duration > 0 else 100
            self.progress_slider.value = 0
            self.progress_display.value = 0

            self.total_time.value = self.format_time(self.duration)
            self.current_time.value = "0:00"
            self.play_pause_btn.icon = ft.Icons.PAUSE

            # استخراج metadata
            song_name = self.extract_song_metadata(song)
            self.song_label.value = song_name
            self.song_label_mini.value = song_name

            self.update_song_list()
            self.update_album_art(song)
            self.highlight_current_song()

            self.save_state()
            logger.info(f"Playing: {song_name}")

        except Exception as e:
            logger.error(f"Play Error: {e}")
            self.song_label.value = f"Error: {str(e)}"
            self.safe_page_update()

    def extract_song_metadata(self, song_path):
        """استخراج متادیتا از آهنگ"""
        try:
            audio = ID3(song_path)
            title = audio.get("TIT2")
            artist = audio.get("TPE1")

            song_name = str(title) if title else os.path.basename(song_path)
            artist_name = str(artist) if artist else "Unknown Artist"

            self.artist_label.value = artist_name
            return song_name
        except Exception as e:
            # اگر متادیتا موجود نبود، از نام فایل استفاده کن
            logger.warning(f"Metadata extraction failed for {os.path.basename(song_path)}: {e}")
            filename = os.path.basename(song_path)
            song_name = os.path.splitext(filename)[0]
            self.artist_label.value = "Unknown Artist"
            return song_name

    def stop_playback(self, _):
        """توقف پخش"""
        if self.playlist:
            try:
                pygame.mixer.music.stop()
                self.is_playing = False
                self._paused = False
                self.play_pause_btn.icon = ft.Icons.PLAY_ARROW
                self.current_position = 0
                self.progress_slider.value = 0
                self.progress_display.value = 0
                self.current_time.value = "0:00"
                self.show_message("Stopped")
                self.safe_page_update()
                logger.info("Music stopped")
            except Exception as e:
                logger.error(f"Stop error: {e}")

    def highlight_current_song(self):
        """Highlight آهنگ جاری"""
        if not self.playlist:
            return

        try:
            self.update_song_list()
            if len(self.playlist) > 10 and self.current_song_index > 8:
                item_height = 52
                scroll_pos = self.current_song_index * item_height
                self.song_list.scroll_to(offset=scroll_pos - 200, duration=300)
        except Exception as e:
            logger.error(f"Highlight error: {e}")

    def play_next(self, _):
        """پخش آهنگ بعدی"""
        if not self.playlist:
            return

        try:
            if self.shuffle_mode:
                if len(self.playlist) == 1:
                    self.current_song_index = 0
                else:
                    old_index = self.current_song_index
                    new_index = random.randint(0, len(self.playlist) - 1)
                    while new_index == old_index and len(self.playlist) > 1:
                        new_index = random.randint(0, len(self.playlist) - 1)
                    self.current_song_index = new_index
            else:
                self.current_song_index = (self.current_song_index + 1) % len(self.playlist)

            self.play_song(self.current_song_index)

        except Exception as e:
            logger.error(f"Play next error: {e}")
            self.show_message(f"Play next error: {str(e)}")

    def play_previous(self, _):
        """پخش آهنگ قبلی"""
        if not self.playlist:
            return

        try:
            if self.shuffle_mode:
                if len(self.playlist) == 1:
                    self.current_song_index = 0
                else:
                    old_index = self.current_song_index
                    new_index = random.randint(0, len(self.playlist) - 1)
                    while new_index == old_index and len(self.playlist) > 1:
                        new_index = random.randint(0, len(self.playlist) - 1)
                    self.current_song_index = new_index
            else:
                self.current_song_index = (self.current_song_index - 1) % len(self.playlist)

            self.play_song(self.current_song_index)

        except Exception as e:
            logger.error(f"Play previous error: {e}")
            self.show_message(f"Play previous error: {str(e)}")

    def on_progress_change(self, e):
        """هنگام تغییر اسلایدر"""
        if self.duration > 0:
            new_position = self.progress_slider.value
            self.current_position = new_position
            self.current_time.value = self.format_time(new_position)
            self.current_time.update()

    def seek_track(self, e):
        """جستجو در آهنگ"""
        if self.playlist and self.duration > 0:
            try:
                new_position = self.progress_slider.value
                self.seek_position = new_position
                self.current_position = new_position
                self.last_update_time = time.time()

                progress_percent = new_position / self.duration if self.duration > 0 else 0
                self.progress_display.value = progress_percent
                self.current_time.value = self.format_time(new_position)

                pygame.mixer.music.stop()
                pygame.mixer.music.play(start=new_position)
                self.is_playing = True
                self.play_pause_btn.icon = ft.Icons.PAUSE

                self.progress_display.update()
                self.current_time.update()
                self.play_pause_btn.update()

            except Exception as e:
                logger.error(f"Seek error: {e}")
                self.show_message(f"Seek Error: {str(e)}")

    def handle_song_end(self):
        """مدیریت پایان آهنگ"""
        self.current_position = 0
        self.seek_position = None

        if self.repeat_mode:
            self.play_song(self.current_song_index)
        else:
            self.play_next(None)

    def adjust_volume(self, e):
        """تنظیم صدا"""
        pygame.mixer.music.set_volume(self.volume_slider.value)
        self.save_state()

    def handle_directory_pick(self, e: ft.FilePickerResultEvent):
        """مدیریت انتخاب دایرکتوری"""
        if e.path:
            self.load_songs(e.path)

    def update_song_list(self):
        """آپدیت لیست آهنگ‌ها با پشتیبانی از جستجو"""
        self.song_list.controls.clear()

        filtered_songs = []
        for idx, song in enumerate(self.playlist):
            song_name = os.path.basename(song).lower()
            if not self.search_query or self.search_query in song_name:
                filtered_songs.append((idx, song))

        for idx, song in filtered_songs:
            is_favorite = song in self.favorite_songs
            is_currently_playing = idx == self.current_song_index

            bg_color = ft.Colors.BLUE_700 if is_currently_playing else ft.Colors.TRANSPARENT
            text_color = ft.Colors.WHITE if is_currently_playing else ft.Colors.WHITE70
            weight = ft.FontWeight.BOLD if is_currently_playing else ft.FontWeight.NORMAL

            favorite_btn = ft.IconButton(
                icon=ft.Icons.FAVORITE if is_favorite else ft.Icons.FAVORITE_BORDER,
                icon_color=ft.Colors.RED_400 if is_favorite else ft.Colors.WHITE70,
                icon_size=20,
                on_click=lambda e, s=song: self.toggle_favorite(s),
            )

            playlist_menu = ft.PopupMenuButton(
                icon=ft.Icons.PLAYLIST_ADD,
                icon_color=ft.Colors.WHITE70,
                icon_size=20,
                items=[
                    ft.PopupMenuItem(
                        text=playlist_name,
                        on_click=lambda e, s=song, p=playlist_name: self.add_song_to_playlist(p, s)
                    ) for playlist_name in self.playlists.keys() if playlist_name != "Favorites"
                ]
            )

            remove_btn = ft.IconButton(
                icon=ft.Icons.DELETE,
                icon_color=ft.Colors.RED_400,
                icon_size=20,
                on_click=lambda e, s=song: self.remove_song_from_playlist(s),
            ) if self.current_playlist_name != "Default" else ft.Container(width=20, height=20)

            song_item = ft.Container(
                content=ft.Row([
                    favorite_btn,
                    ft.Text(
                        f"{idx + 1}. {os.path.basename(song)}",
                        color=text_color,
                        weight=weight,
                        size=12,
                        expand=True,
                        overflow=ft.TextOverflow.ELLIPSIS,
                    ),
                    playlist_menu,
                    remove_btn,
                ],
                    alignment=ft.MainAxisAlignment.START,
                    vertical_alignment=ft.CrossAxisAlignment.CENTER,
                    spacing=5,
                ),
                bgcolor=bg_color,
                border_radius=8,
                padding=8,
                on_click=lambda e, i=idx: self.on_song_click(i),
                height=50,
            )
            self.song_list.controls.append(song_item)

        self.safe_page_update()

    def on_song_click(self, index):
        """مدیریت کلیک روی آهنگ"""
        self.play_song(index)

    def toggle_playback(self, _):
        """تغییر حالت پخش/توقف"""
        if not self.playlist:
            return

        try:
            if self.is_playing:
                pygame.mixer.music.pause()
                self.is_playing = False
                self._paused = True
                self.play_pause_btn.icon = ft.Icons.PLAY_ARROW
                self.show_message("Paused")
                logger.info("Music paused")
            else:
                if self._paused:
                    pygame.mixer.music.unpause()
                    self.is_playing = True
                    self.play_pause_btn.icon = ft.Icons.PAUSE
                    self._paused = False
                    self.show_message("Resumed")
                    logger.info("Music resumed")
                else:
                    self.play_song(self.current_song_index)
                    return

            self.safe_page_update()

        except Exception as e:
            logger.error(f"Playback toggle error: {e}")
            self.show_message(f"Playback error: {str(e)}")

    def show_message(self, text):
        """نمایش پیام"""
        try:
            self.page.snack_bar = ft.SnackBar(
                content=ft.Text(text, color=ft.Colors.WHITE),
                bgcolor=ft.Colors.BLUE_700,
                duration=3000,
            )
            self.page.snack_bar.open = True
            self.safe_page_update()
        except Exception as e:
            logger.error(f"Show message error: {e}")

    def format_time(self, seconds):
        """فرمت زمان"""
        if seconds < 0:
            return "0:00"
        minutes = int(seconds // 60)
        secs = int(seconds % 60)
        return f"{minutes}:{secs:02d}"

    def update_album_art(self, song_path):
        """آپدیت آلبوم آرت برای هر دو حالت"""
        try:
            # اول از کش بررسی کن
            cached_art = self.cache_manager.load_album_art(song_path)
            if cached_art:
                img_data = base64.b64encode(cached_art).decode("utf-8")
                self.album_image.src_base64 = img_data
                # آپدیت برای mini-player
                if hasattr(self, 'album_image_mini'):
                    self.album_image_mini.src_base64 = img_data
                return

            # اگر در کش نبود، از فایل بخوان
            tag_found = False
            try:
                audio = ID3(song_path)
                for tag in audio.values():
                    if isinstance(tag, APIC):
                        # در کش ذخیره کن
                        self.cache_manager.save_album_art(song_path, tag.data)
                        img_data = base64.b64encode(tag.data).decode("utf-8")
                        self.album_image.src_base64 = img_data
                        # آپدیت برای mini-player
                        if hasattr(self, 'album_image_mini'):
                            self.album_image_mini.src_base64 = img_data
                        tag_found = True
                        break
            except Exception as id3_error:
                logger.warning(f"ID3 tag not found for {os.path.basename(song_path)}: {id3_error}")

            # اگر تگ ID3 پیدا نشد، از تصویر پیش‌فرض استفاده کن
            if not tag_found:
                self.album_image.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"
                if hasattr(self, 'album_image_mini'):
                    self.album_image_mini.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"

        except Exception as e:
            logger.error(f"Album art error: {e}")
            self.album_image.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"
            if hasattr(self, 'album_image_mini'):
                self.album_image_mini.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"

        self.safe_page_update()

    def toggle_repeat(self, _):
        """تغییر حالت تکرار"""
        self.repeat_mode = not self.repeat_mode
        self.repeat_btn.icon_color = ft.Colors.BLUE_400 if self.repeat_mode else ft.Colors.WHITE70
        self.safe_page_update()
        self.save_state()

    def toggle_shuffle(self, _):
        """تغییر حالت تصادفی"""
        self.shuffle_mode = not self.shuffle_mode
        self.shuffle_btn.icon_color = ft.Colors.BLUE_400 if self.shuffle_mode else ft.Colors.WHITE70
        self.safe_page_update()
        self.save_state()

    def delete_current_playlist(self, e):
        """حذف پلی‌لیست جاری"""
        if self.current_playlist_name in ["Default", "Favorites"]:
            self.show_message("Cannot delete Default or Favorites playlist!")
            return

        def confirm_delete(e):
            del self.playlists[self.current_playlist_name]
            self.current_playlist_name = "Default"
            self.playlist = self.playlists["Default"].copy()
            self.current_song_index = 0
            self.update_song_list()
            self.update_song_list_header()
            if self.playlist:
                self.play_song(0)
            else:
                self.song_label.value = "No songs in playlist"
                self.album_image.src = "https://cdn-icons-png.flaticon.com/512/727/727240.png"
                self.is_playing = False
                self.play_pause_btn.icon = ft.Icons.PLAY_ARROW
            self.save_playlists()
            self.page.close(dlg)
            self.show_message(f"Playlist deleted!")
            logger.info(f"Playlist deleted: {self.current_playlist_name}")

        dlg = ft.AlertDialog(
            title=ft.Text("Delete Playlist", color=ft.Colors.WHITE),
            content=ft.Text(f"Are you sure you want to delete '{self.current_playlist_name}'?", color=ft.Colors.WHITE),
            actions=[
                ft.TextButton("Cancel", on_click=lambda e: self.page.close(dlg),
                              style=ft.ButtonStyle(color=ft.Colors.WHITE)),
                ft.TextButton("Delete", on_click=confirm_delete, style=ft.ButtonStyle(color=ft.Colors.RED_400)),
            ],
            bgcolor=ft.Colors.BLUE_900,
        )
        self.page.open(dlg)

    def add_song_to_playlist(self, playlist_name, song_path):
        """اضافه کردن آهنگ به پلی‌لیست"""
        if playlist_name in self.playlists and song_path not in self.playlists[playlist_name]:
            self.playlists[playlist_name].append(song_path)
            self.save_playlists()
            self.show_message(f"Song added to {playlist_name}!")
        elif song_path in self.playlists[playlist_name]:
            self.show_message("Song already in playlist!")

    def add_songs_to_playlist(self, playlist_name, songs):
        """اضافه کردن چند آهنگ به پلی‌لیست"""
        if playlist_name not in self.playlists:
            self.playlists[playlist_name] = []

        for song in songs:
            if song not in self.playlists[playlist_name]:
                self.playlists[playlist_name].append(song)

        self.save_playlists()

    def remove_song_from_playlist(self, song_path):
        """حذف آهنگ از پلی‌لیست"""
        if self.current_playlist_name in self.playlists and song_path in self.playlists[self.current_playlist_name]:
            if song_path == self.playlist[self.current_song_index]:
                if len(self.playlist) > 1:
                    self.play_next(None)
                else:
                    self.is_playing = False
                    self.play_pause_btn.icon = ft.Icons.PLAY_ARROW

            self.playlists[self.current_playlist_name].remove(song_path)
            self.playlist = self.playlists[self.current_playlist_name].copy()
            self.current_song_index = min(self.current_song_index, len(self.playlist) - 1) if self.playlist else 0
            self.update_song_list()
            self.save_playlists()
            self.show_message("Song removed from playlist!")

    def toggle_favorite(self, song_path):
        """تغییر حالت علاقه‌مندی"""
        if song_path in self.favorite_songs:
            self.favorite_songs.remove(song_path)
            if self.favorites_playlist in self.playlists and song_path in self.playlists[self.favorites_playlist]:
                self.playlists[self.favorites_playlist].remove(song_path)
            self.show_message("Removed from favorites!")
        else:
            self.favorite_songs.add(song_path)
            if self.favorites_playlist not in self.playlists:
                self.playlists[self.favorites_playlist] = []
            if song_path not in self.playlists[self.favorites_playlist]:
                self.playlists[self.favorites_playlist].append(song_path)
            self.show_message("Added to favorites!")

        self.update_song_list()
        self.save_playlists()

    def save_playlists(self):
        """ذخیره پلی‌لیست‌ها"""
        try:
            data = {
                "playlists": self.playlists,
                "favorites": list(self.favorite_songs),
                "last_modified": datetime.now().isoformat()
            }
            with open(PLAYLISTS_FILE, "w", encoding="utf-8") as f:
                json.dump(data, f, ensure_ascii=False, indent=2)
        except Exception as e:
            logger.error(f"Save playlists error: {e}")

    def save_state(self):
        """ذخیره وضعیت"""
        try:
            state = {
                "playlist": self.playlist,
                "current_song_index": self.current_song_index,
                "shuffle_mode": self.shuffle_mode,
                "repeat_mode": self.repeat_mode,
                "volume": self.volume_slider.value if hasattr(self, "volume_slider") else 0.5,
                "current_playlist": self.current_playlist_name,
                "theme": self.current_theme,
                "last_position": self.current_position,
                "save_time": datetime.now().isoformat()
            }
            with open(SETTINGS_FILE, "w", encoding="utf-8") as f:
                json.dump(state, f, ensure_ascii=False, indent=2)
        except Exception as e:
            logger.error(f"Save state error: {e}")

    def load_state(self):
        """بارگذاری وضعیت"""
        if not os.path.exists(SETTINGS_FILE):
            return
        try:
            with open(SETTINGS_FILE, "r", encoding="utf-8") as f:
                state = json.load(f)

            self.playlist = state.get("playlist", [])
            self.current_song_index = min(state.get("current_song_index", 0),
                                          len(self.playlist) - 1) if self.playlist else 0
            self.shuffle_mode = state.get("shuffle_mode", False)
            self.repeat_mode = state.get("repeat_mode", False)
            volume = state.get("volume", 0.5)
            self.current_playlist_name = state.get("current_playlist", "Default")
            self.current_theme = state.get("theme", "Blue Night")
            last_position = state.get("last_position", 0)

            self.shuffle_btn.icon_color = ft.Colors.BLUE_400 if self.shuffle_mode else ft.Colors.WHITE70
            self.repeat_btn.icon_color = ft.Colors.BLUE_400 if self.repeat_mode else ft.Colors.WHITE70
            self.volume_slider.value = volume
            pygame.mixer.music.set_volume(volume)

            self.page_container.gradient = ft.LinearGradient(
                begin=ft.alignment.top_left,
                end=ft.alignment.bottom_right,
                colors=self.themes[self.current_theme]
            )

            self.update_song_list()

            if self.playlist:
                current_song = self.playlist[self.current_song_index]
                self.song_label.value = self.extract_song_metadata(current_song)
                try:
                    audio = MP3(current_song)
                    self.duration = audio.info.length
                    self.progress_slider.max = self.duration
                    self.total_time.value = self.format_time(self.duration)
                    self.progress_slider.value = last_position
                    self.current_position = last_position
                except Exception as e:
                    logger.error(f"Duration read error: {e}")
                    self.duration = 0
                    self.progress_slider.max = 100
                    self.total_time.value = "00:00"
                    self.progress_slider.value = 0

                self.update_album_art(current_song)
                self.is_playing = False
                self.play_pause_btn.icon = ft.Icons.PLAY_ARROW

                self.scroll_to_current_song()

            self.loaded_from_state = True
            self.state_loaded = True
            self.safe_page_update()
        except Exception as e:
            logger.error(f"Load state error: {e}")

    def safe_page_update(self):
        """آپدیت امن صفحه"""
        try:
            if self.page and hasattr(self.page, 'update'):
                self.page.update()
        except Exception as e:
            if "socket" not in str(e):
                logger.error(f"Page update error: {e}")

    def dispose(self):
        """تمیز کردن منابع"""
        self.stop_progress_thread = True
        self.running = False

        if hasattr(self, 'progress_timer') and self.progress_timer:
            self.progress_timer.cancel()

        try:
            pygame.mixer.music.stop()
            pygame.mixer.music.unload()
            pygame.mixer.quit()
        except:
            pass

        self.save_state()
        self.save_playlists()

        try:
            self.page.window_destroy()
        except:
            pass

        logger.info("Application disposed successfully")


def main(page: ft.Page):
    """تابع اصلی برنامه"""
    try:
        MusicPlayerApp(page)
    except Exception as e:
        logger.critical(f"Application failed to start: {e}")
        raise


if __name__ == "__main__":
    ft.app(target=main)
