import flet as ft
import requests
import os
import json
from urllib.parse import urlencode
import time
from PIL import Image, ImageEnhance, ImageFilter
import io
import base64

# مسیر فایل تنظیمات
SETTINGS_FILE = "gallery_settings.json"


def load_settings():
    """بارگذاری تنظیمات از فایل"""
    default_settings = {
        "theme_mode": "light",
        "primary_color": "blue",
        "image_quality": "largeImageURL",
        "download_quality": "largeImageURL",
        "images_per_page": 12,
        "enable_cache": True,
        # تنظیمات جدید بهبود کیفیت
        "default_enhancement": "none",
        "default_enhancement_level": "medium",
        "default_upscale": "none"
    }

    try:
        if os.path.exists(SETTINGS_FILE):
            with open(SETTINGS_FILE, 'r', encoding='utf-8') as f:
                loaded_settings = json.load(f)
                # مطمئن شوید همه کلیدهای ضروری وجود دارند
                for key in default_settings:
                    if key not in loaded_settings:
                        loaded_settings[key] = default_settings[key]
                return loaded_settings
    except Exception as e:
        print(f"خطا در بارگذاری تنظیمات: {e}")

    return default_settings


def save_settings(settings):
    """ذخیره تنظیمات در فایل"""
    try:
        with open(SETTINGS_FILE, 'w', encoding='utf-8') as f:
            json.dump(settings, f, ensure_ascii=False, indent=4)
        return True
    except Exception as e:
        print(f"خطا در ذخیره تنظیمات: {e}")
        return False


def enhance_image_local(image_data, enhancement_type, enhancement_level, upscale_factor):
    """
    بهبود کیفیت تصویر با استفاده از Pillow
    """
    try:
        # باز کردن تصویر از داده‌های باینری
        image = Image.open(io.BytesIO(image_data))

        # ذخیره اطلاعات اصلی
        original_format = image.format
        original_size = image.size

        # تنظیمات سطح بهبود
        level_settings = {
            "low": {"sharpness": 1.3, "contrast": 1.2, "brightness": 1.1, "color": 1.1},
            "medium": {"sharpness": 1.6, "contrast": 1.4, "brightness": 1.2, "color": 1.2},
            "high": {"sharpness": 2.0, "contrast": 1.6, "brightness": 1.3, "color": 1.3},
            "ultra": {"sharpness": 2.5, "contrast": 1.8, "brightness": 1.4, "color": 1.4}
        }

        settings = level_settings.get(enhancement_level, level_settings["medium"])

        # افزایش رزولوشن اگر انتخاب شده
        if upscale_factor != "none":
            if upscale_factor == "2x":
                new_size = (original_size[0] * 2, original_size[1] * 2)
            elif upscale_factor == "4x":
                new_size = (original_size[0] * 4, original_size[1] * 4)
            else:
                new_size = original_size

            # استفاده از الگوریتم با کیفیت بالا برای افزایش سایز
            image = image.resize(new_size, Image.Resampling.LANCZOS)

            # اعمال فیلتر شارپنس پس از افزایش سایز
            image = image.filter(ImageFilter.UnsharpMask(radius=2, percent=150, threshold=3))

        # اعمال فیلتر شارپنس اولیه برای همه حالت‌ها
        if enhancement_type != "none":
            image = image.filter(ImageFilter.UnsharpMask(radius=2, percent=100, threshold=3))

        # اعمال بهبودها بر اساس نوع انتخاب شده
        if enhancement_type != "none":
            if enhancement_type in ["sharpness", "all"]:
                enhancer = ImageEnhance.Sharpness(image)
                image = enhancer.enhance(settings["sharpness"])

            if enhancement_type in ["contrast", "all"]:
                enhancer = ImageEnhance.Contrast(image)
                image = enhancer.enhance(settings["contrast"])

            if enhancement_type in ["brightness", "all"]:
                enhancer = ImageEnhance.Brightness(image)
                image = enhancer.enhance(settings["brightness"])

            if enhancement_type in ["color", "all"]:
                enhancer = ImageEnhance.Color(image)
                image = enhancer.enhance(settings["color"])

        # تبدیل به داده‌های باینری با کیفیت بالا
        output_buffer = io.BytesIO()

        if original_format and original_format.lower() in ['png', 'gif']:
            image.save(output_buffer, format=original_format, optimize=True)
        else:
            image.save(output_buffer, format='JPEG', quality=95, optimize=True, subsampling=0)

        enhanced_data = output_buffer.getvalue()

        return enhanced_data, original_size, image.size

    except Exception as e:
        print(f"خطا در بهبود کیفیت تصویر: {e}")
        return image_data, (0, 0), (0, 0)


def main(page: ft.Page):
    # بارگذاری تنظیمات ذخیره شده
    saved_settings = load_settings()

    # تنظیمات اولیه صفحه
    page.title = "گالری آنلاین"
    page.window.width = 1000
    page.window.height = 700
    page.window.min_width = 800
    page.window.min_height = 600
    page.padding = 0

    # تنظیم تم بر اساس تنظیمات ذخیره شده
    theme_mode = ft.ThemeMode.LIGHT if saved_settings["theme_mode"] == "light" else ft.ThemeMode.DARK
    page.theme_mode = theme_mode

    # متغیرهای حالت برنامه
    current_page_num = 1
    is_loading = False
    selected_category = None
    selected_hit = None
    selected_resolution = saved_settings["image_quality"]
    previous_categories = []

    # کش تصاویر
    image_cache = {}
    CACHE_DURATION = 3600  # 1 hour

    # تنظیمات کاربر - با مقادیر ذخیره شده
    user_settings = saved_settings.copy()

    # رنگ‌های تم
    color_themes = {
        "آبی": ft.Colors.BLUE,
        "سبز": ft.Colors.GREEN,
        "بنفش": ft.Colors.PURPLE,
        "نارنجی": ft.Colors.ORANGE,
        "قرمز": ft.Colors.RED,
        "صورتی": ft.Colors.PINK
    }

    # نگاشت نام رنگ به مقدار ذخیره سازی
    color_name_to_value = {
        "آبی": "blue",
        "سبز": "green",
        "بنفش": "purple",
        "نارنجی": "orange",
        "قرمز": "red",
        "صورتی": "pink"
    }

    # نگاشت مقدار ذخیره سازی به نام فارسی
    color_value_to_name = {
        "blue": "آبی",
        "green": "سبز",
        "purple": "بنفش",
        "orange": "نارنجی",
        "red": "قرمز",
        "pink": "صورتی"
    }

    # تبدیل نام رنگ ذخیره شده به شیء رنگ Flet
    def get_color_from_setting(color_name):
        color_map = {
            "blue": ft.Colors.BLUE,
            "green": ft.Colors.GREEN,
            "purple": ft.Colors.PURPLE,
            "orange": ft.Colors.ORANGE,
            "red": ft.Colors.RED,
            "pink": ft.Colors.PINK
        }
        return color_map.get(color_name, ft.Colors.BLUE)

    # تبدیل شیء رنگ Flet به نام برای ذخیره سازی
    def get_color_name(color_obj):
        reverse_map = {
            ft.Colors.BLUE: "blue",
            ft.Colors.GREEN: "green",
            ft.Colors.PURPLE: "purple",
            ft.Colors.ORANGE: "orange",
            ft.Colors.RED: "red",
            ft.Colors.PINK: "pink"
        }
        return reverse_map.get(color_obj, "blue")

    # دسته‌بندی‌های واقعی با آیکون‌های معتبر
    categories = [
        # طبیعت و محیط زیست
        {"name": "طبیعت", "icon": ft.Icons.NATURE, "query": "nature"},
        {"name": "منظره", "icon": ft.Icons.LANDSCAPE, "query": "landscape"},
        {"name": "کوهستان", "icon": ft.Icons.TERRAIN, "query": "mountain"},
        {"name": "دریا", "icon": ft.Icons.WATER, "query": "ocean"},
        {"name": "جنگل", "icon": ft.Icons.PARK, "query": "forest"},
        {"name": "گل‌ها", "icon": ft.Icons.LOCAL_FLORIST, "query": "flowers"},
        {"name": "آسمان", "icon": ft.Icons.CLOUD, "query": "sky"},
        {"name": "درختان", "icon": ft.Icons.PARK, "query": "trees"},

        # حیوانات
        {"name": "حیوانات", "icon": ft.Icons.PETS, "query": "animals"},
        {"name": "گربه‌ها", "icon": ft.Icons.PETS, "query": "cats"},
        {"name": "سگ‌ها", "icon": ft.Icons.PETS, "query": "dogs"},
        {"name": "حیوانات وحشی", "icon": ft.Icons.PETS, "query": "wildlife"},
        {"name": "پرندگان", "icon": ft.Icons.PETS, "query": "birds"},
        {"name": "اسب", "icon": ft.Icons.PETS, "query": "horse"},
        {"name": "پروانه", "icon": ft.Icons.PETS, "query": "butterfly"},

        # شهرها و معماری
        {"name": "شهرها", "icon": ft.Icons.LOCATION_CITY, "query": "cities"},
        {"name": "ساختمان‌ها", "icon": ft.Icons.APARTMENT, "query": "buildings"},
        {"name": "خیابان‌ها", "icon": ft.Icons.TRAFFIC, "query": "streets"},
        {"name": "خانه", "icon": ft.Icons.HOUSE, "query": "house"},
        {"name": "پل", "icon": ft.Icons.ACCOUNT_BALANCE, "query": "bridge"},

        # هنر و خلاقیت
        {"name": "هنر", "icon": ft.Icons.PALETTE, "query": "art"},
        {"name": "نقاشی", "icon": ft.Icons.BRUSH, "query": "painting"},
        {"name": "رنگ", "icon": ft.Icons.FORMAT_COLOR_FILL, "query": "colorful"},
        {"name": "خلاقیت", "icon": ft.Icons.LIGHTBULB, "query": "creative"},
        {"name": "طرح", "icon": ft.Icons.DRAW, "query": "pattern"},

        # ورزش
        {"name": "ورزش", "icon": ft.Icons.SPORTS, "query": "sports"},
        {"name": "فوتبال", "icon": ft.Icons.SPORTS_FOOTBALL, "query": "football"},
        {"name": "کوهنوردی", "icon": ft.Icons.DIRECTIONS_WALK, "query": "hiking"},
        {"name": "دوچرخه‌سواری", "icon": ft.Icons.DIRECTIONS_BIKE, "query": "cycling"},
        {"name": "شنا", "icon": ft.Icons.POOL, "query": "swimming"},

        # غذا
        {"name": "غذا", "icon": ft.Icons.RESTAURANT, "query": "food"},
        {"name": "میوه", "icon": ft.Icons.APPLE, "query": "fruits"},
        {"name": "قهوه", "icon": ft.Icons.COFFEE, "query": "coffee"},
        {"name": "دسر", "icon": ft.Icons.CAKE, "query": "dessert"},
        {"name": "پیتزا", "icon": ft.Icons.LOCAL_PIZZA, "query": "pizza"},

        # مد و زیبایی
        {"name": "مد", "icon": ft.Icons.STYLE, "query": "fashion"},
        {"name": "لباس", "icon": ft.Icons.CHECKROOM, "query": "clothing"},
        {"name": "زیبایی", "icon": ft.Icons.FACE, "query": "beauty"},

        # سفر
        {"name": "سفر", "icon": ft.Icons.FLIGHT, "query": "travel"},
        {"name": "ماجراجویی", "icon": ft.Icons.EXPLORE, "query": "adventure"},

        # تکنولوژی
        {"name": "تکنولوژی", "icon": ft.Icons.COMPUTER, "query": "technology"},
        {"name": "کدنویسی", "icon": ft.Icons.CODE, "query": "coding"},
        {"name": "ربات", "icon": ft.Icons.SMART_TOY, "query": "robot"},
        {"name": "فضا", "icon": ft.Icons.STAR, "query": "space"},

        # فرهنگ و هنر
        {"name": "کتاب", "icon": ft.Icons.BOOK, "query": "books"},
        {"name": "فیلم", "icon": ft.Icons.MOVIE, "query": "movies"},
        {"name": "موسیقی", "icon": ft.Icons.MUSIC_NOTE, "query": "music"},
        {"name": "سینما", "icon": ft.Icons.THEATER_COMEDY, "query": "cinema"},

        # کارتون و انیمیشن
        {"name": "کارتون", "icon": ft.Icons.ANIMATION, "query": "cartoon"},
        {"name": "انیمیشن", "icon": ft.Icons.ANIMATION, "query": "animation"},
        {"name": "شخصیت‌های کارتونی", "icon": ft.Icons.PERSON, "query": "cartoon+characters"},
        {"name": "کمیک", "icon": ft.Icons.MENU_BOOK, "query": "comic"},
        {"name": "فانتزی", "icon": ft.Icons.AUTO_AWESOME, "query": "fantasy"},
        {"name": "بازی‌های ویدیویی", "icon": ft.Icons.SPORTS_ESPORTS, "query": "video+games"},
        {"name": "قهرمان", "icon": ft.Icons.SECURITY, "query": "superhero"},
        {"name": "خنده‌دار", "icon": ft.Icons.SENTIMENT_SATISFIED, "query": "funny+cartoon"},

        # فصل‌ها
        {"name": "بهار", "icon": ft.Icons.NATURE, "query": "spring"},
        {"name": "تابستان", "icon": ft.Icons.WB_SUNNY, "query": "summer"},
        {"name": "پاییز", "icon": ft.Icons.PARK, "query": "autumn"},
        {"name": "زمستان", "icon": ft.Icons.AC_UNIT, "query": "winter"},
        {"name": "برف", "icon": ft.Icons.AC_UNIT, "query": "snow"},

        # مناسبت‌ها
        {"name": "عروسی", "icon": ft.Icons.FAVORITE, "query": "wedding"},
        {"name": "جشن", "icon": ft.Icons.CELEBRATION, "query": "celebration"},
        {"name": "عشق", "icon": ft.Icons.FAVORITE, "query": "love"},

        # وسایل نقلیه
        {"name": "ماشین‌ها", "icon": ft.Icons.DIRECTIONS_CAR, "query": "cars"},
        {"name": "موتورسیکلت", "icon": ft.Icons.MOTORCYCLE, "query": "motorcycle"},
        {"name": "قایق", "icon": ft.Icons.DIRECTIONS_BOAT, "query": "boats"},
        {"name": "هواپیما", "icon": ft.Icons.FLIGHT, "query": "airplanes"},

        # سایر دسته‌بندی‌های واقعی
        {"name": "پس‌زمینه", "icon": ft.Icons.WALLPAPER, "query": "background"},
        {"name": "تاریخی", "icon": ft.Icons.HISTORY, "query": "history"},
        {"name": "علمی", "icon": ft.Icons.SCIENCE, "query": "science"},
        {"name": "آموزشی", "icon": ft.Icons.SCHOOL, "query": "education"},
        {"name": "بازی", "icon": ft.Icons.SPORTS_ESPORTS, "query": "games"},
        {"name": "سلامتی", "icon": ft.Icons.HEALTH_AND_SAFETY, "query": "health"},
        {"name": "تجاری", "icon": ft.Icons.BUSINESS, "query": "business"},
        {"name": "کار", "icon": ft.Icons.WORK, "query": "work"},
        {"name": "خانواده", "icon": ft.Icons.FAMILY_RESTROOM, "query": "family"},
        {"name": "دوستان", "icon": ft.Icons.GROUP, "query": "friends"},
    ]

    # بررسی اتصال اینترنت
    def check_internet():
        try:
            requests.get("https://www.google.com", timeout=5)
            return True
        except requests.ConnectionError:
            return False

    # سیستم کش برای تصاویر
    def get_cached_image(url):
        if not user_settings["enable_cache"]:
            return download_image_data(url)

        current_time = time.time()
        if url in image_cache:
            cached_time, data = image_cache[url]
            if current_time - cached_time < CACHE_DURATION:
                return data

        # دانلود و کش کردن تصویر
        data = download_image_data(url)
        if data:
            image_cache[url] = (current_time, data)
        return data

    def download_image_data(url):
        try:
            response = requests.get(url, timeout=30)
            response.raise_for_status()
            return response.content
        except requests.exceptions.RequestException:
            return None

    def clear_old_cache():
        current_time = time.time()
        expired_urls = [
            url for url, (cached_time, _) in image_cache.items()
            if current_time - cached_time > CACHE_DURATION
        ]
        for url in expired_urls:
            del image_cache[url]

    # دریافت تصاویر از Pixabay
    def fetch_images(query, page_num=1, limit=None):
        if limit is None:
            limit = user_settings["images_per_page"]

        if not check_internet():
            show_snackbar("اتصال اینترنت وجود ندارد.")
            return []

        params = {
            "key": "46241976-2e7bc03d01849a4fb1a40b1a8",
            "q": query,
            "image_type": "photo",
            "page": page_num,
            "per_page": limit
        }
        pixabay_url = f"https://pixabay.com/api/?{urlencode(params)}"

        try:
            response = requests.get(pixabay_url, timeout=50)
            response.raise_for_status()
            data = response.json()
            if "hits" in data:
                return data["hits"][:limit]
            else:
                show_snackbar("تصویری یافت نشد.")
                return []
        except requests.exceptions.RequestException as e:
            show_snackbar(f"خطا در دریافت تصاویر از Pixabay: {e}")
            return []

    # نمایش اسنک‌بار - روش صحیح در Flet
    def show_snackbar(message):
        page.snack_bar = ft.SnackBar(
            content=ft.Text(message),
            action="باشه"
        )
        page.snack_bar.open = True
        page.update()

    # ایجاد کنترل تصویر
    def create_image_control(hit):
        img_url = hit.get(selected_resolution, hit["largeImageURL"])

        # اطلاعات تصویر
        tags = hit.get("tags", "").split(",")
        user = hit.get("user", "ناشناس")
        likes = hit.get("likes", 0)

        # استفاده از کش برای تصویر
        image_control = ft.Image(
            src=img_url,
            width=200,
            height=150,
            fit=ft.ImageFit.COVER,
            gapless_playback=False,
            tooltip="کلیک برای بزرگنمایی"
        )

        # دکمه‌های عملیات - با رفع مشکل event handlerها
        def on_zoom_click(e, hit=hit):
            show_zoom_dialog(hit)

        def on_download_click(e, hit=hit):
            download_image(hit)

        def on_info_click(e, hit=hit):
            show_image_info(hit)

        zoom_icon = ft.IconButton(
            icon=ft.Icons.ZOOM_IN,
            icon_color=ft.Colors.WHITE,
            on_click=on_zoom_click,
            tooltip="بزرگنمایی"
        )

        download_icon = ft.IconButton(
            icon=ft.Icons.DOWNLOAD,
            icon_color=ft.Colors.WHITE,
            on_click=on_download_click,
            tooltip="دانلود"
        )

        info_icon = ft.IconButton(
            icon=ft.Icons.INFO_OUTLINE,
            icon_color=ft.Colors.WHITE,
            on_click=on_info_click,
            tooltip="اطلاعات تصویر"
        )

        # کانتینر تصویر
        def on_image_click(e, hit=hit):
            show_zoom_dialog(hit)

        image_container = ft.Container(
            content=ft.Stack(
                [
                    image_control,
                    ft.Container(
                        content=ft.Row(
                            [info_icon, download_icon, zoom_icon],
                            alignment=ft.MainAxisAlignment.END
                        ),
                        bgcolor=ft.Colors.BLACK54,
                        border_radius=ft.border_radius.only(top_left=10, top_right=10),
                        padding=5,
                        top=0,
                        right=0,
                        left=0
                    ),
                    ft.Container(
                        content=ft.Column(
                            [
                                ft.Text(
                                    f"❤️ {likes}",
                                    size=12,
                                    color=ft.Colors.WHITE,
                                    weight=ft.FontWeight.BOLD
                                ),
                                ft.Text(
                                    f"توسط {user}",
                                    size=10,
                                    color=ft.Colors.WHITE
                                )
                            ],
                            spacing=2
                        ),
                        bgcolor=ft.Colors.BLACK54,
                        border_radius=ft.border_radius.only(bottom_left=10, bottom_right=10),
                        padding=5,
                        bottom=0,
                        right=0,
                        left=0
                    )
                ]
            ),
            width=220,
            height=180,
            border_radius=10,
            shadow=ft.BoxShadow(
                blur_radius=15,
                color=ft.Colors.BLACK38,
                offset=ft.Offset(0, 5)
            ),
            margin=5,
            on_click=on_image_click
        )

        return image_container

    def show_zoom_dialog(hit):
        img_url = hit.get("largeImageURL", hit.get("webformatURL", ""))

        if not img_url:
            show_snackbar("آدرس تصویر نامعتبر است")
            return

        # ایجاد تصویر با قابلیت اسکرول اگر بزرگتر از صفحه باشد
        zoomed_image = ft.Image(
            src=img_url,
            width=700,
            height=400,
            fit=ft.ImageFit.CONTAIN,
            gapless_playback=True
        )

        # اطلاعات تصویر
        tags = hit.get("tags", "").split(",")
        user = hit.get("user", "ناشناس")
        likes = hit.get("likes", 0)
        views = hit.get("views", 0)
        downloads = hit.get("downloads", 0)
        comments = hit.get("comments", 0)

        def close_dialog(e):
            page.overlay.remove(zoom_overlay_container)
            page.update()

        def download_from_dialog(e):
            download_image(hit)
            close_dialog(e)

        # ایجاد کنترل‌های برچسب‌ها با Row به جای Wrap
        tag_controls = []
        for tag in tags[:8]:
            if tag.strip():
                tag_controls.append(
                    ft.Container(
                        content=ft.Text(tag.strip(), size=12),
                        padding=ft.padding.symmetric(horizontal=8, vertical=4),
                        bgcolor=ft.Colors.BLUE_100,
                        border_radius=15,
                        margin=2
                    )
                )

        # محتوای اصلی
        content_column = ft.Column([
            # هدر
            ft.Row([
                ft.Text("بزرگنمایی تصویر",
                        size=20,
                        weight=ft.FontWeight.BOLD,
                        expand=True),
                ft.IconButton(
                    icon=ft.Icons.CLOSE,
                    on_click=close_dialog,
                    tooltip="بستن"
                )
            ], alignment=ft.MainAxisAlignment.SPACE_BETWEEN),

            ft.Divider(height=10),

            # تصویر
            ft.Container(
                content=zoomed_image,
                alignment=ft.alignment.center,
                padding=10,
                bgcolor=ft.Colors.BLACK12 if page.theme_mode == ft.ThemeMode.LIGHT else ft.Colors.GREY_900,
                border_radius=10
            ),

            ft.Divider(height=15),

            # اطلاعات تصویر
            ft.Container(
                content=ft.Column([
                    # ردیف اول اطلاعات
                    ft.Row([
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.PERSON, size=16, color=ft.Colors.BLUE),
                                ft.Text("عکاس", size=12, color=ft.Colors.GREY_600),
                                ft.Text(user, size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                        ft.VerticalDivider(width=1),
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.FAVORITE, size=16, color=ft.Colors.RED),
                                ft.Text("لایک", size=12, color=ft.Colors.GREY_600),
                                ft.Text(str(likes), size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                        ft.VerticalDivider(width=1),
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.REMOVE_RED_EYE, size=16, color=ft.Colors.GREEN),
                                ft.Text("بازدید", size=12, color=ft.Colors.GREY_600),
                                ft.Text(str(views), size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                    ], alignment=ft.MainAxisAlignment.SPACE_AROUND),

                    # ردیف دوم اطلاعات
                    ft.Row([
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.DOWNLOAD, size=16, color=ft.Colors.ORANGE),
                                ft.Text("دانلود", size=12, color=ft.Colors.GREY_600),
                                ft.Text(str(downloads), size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                        ft.VerticalDivider(width=1),
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.COMMENT, size=16, color=ft.Colors.PURPLE),
                                ft.Text("نظر", size=12, color=ft.Colors.GREY_600),
                                ft.Text(str(comments), size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                        ft.VerticalDivider(width=1),
                        ft.Container(
                            content=ft.Column([
                                ft.Icon(ft.Icons.TAG, size=16, color=ft.Colors.TEAL),
                                ft.Text("برچسب", size=12, color=ft.Colors.GREY_600),
                                ft.Text(str(len(tags)), size=14, weight=ft.FontWeight.BOLD),
                            ], horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                            padding=5,
                            expand=True
                        ),
                    ], alignment=ft.MainAxisAlignment.SPACE_AROUND),

                    ft.Divider(height=10),

                    # برچسب‌ها - با Row و wrap
                    ft.Container(
                        content=ft.Column([
                            ft.Text("برچسب‌ها:", size=14, weight=ft.FontWeight.BOLD),
                            ft.Container(
                                content=ft.Row(
                                    controls=tag_controls,
                                    wrap=True,
                                    spacing=5,
                                    run_spacing=5
                                ),
                                padding=10
                            )
                        ]),
                        padding=10
                    )
                ]),
                bgcolor=ft.Colors.CYAN_100,
                border_radius=10,
                padding=10
            ),

            ft.Divider(height=15),

            # دکمه‌های عمل
            ft.Row([
                ft.ElevatedButton(
                    "دانلود تصویر",
                    icon=ft.Icons.DOWNLOAD,
                    on_click=download_from_dialog,
                    style=ft.ButtonStyle(
                        color=ft.Colors.WHITE,
                        bgcolor=ft.Colors.GREEN,
                        padding=20
                    ),
                    expand=True
                ),
                ft.TextButton(
                    "بستن",
                    icon=ft.Icons.CLOSE,
                    on_click=close_dialog,
                    style=ft.ButtonStyle(padding=20),
                    expand=True
                )
            ], spacing=10)
        ], scroll=ft.ScrollMode.ADAPTIVE)

        # کانتینر اصلی
        zoom_container = ft.Container(
            content=content_column,
            width=750,
            height=650,
            bgcolor=ft.Colors.SURFACE,
            border_radius=15,
            padding=20,
            shadow=ft.BoxShadow(
                blur_radius=50,
                color=ft.Colors.BLACK54,
                offset=ft.Offset(0, 5)
            )
        )

        # کانتینر overlay برای مرکز کردن
        zoom_overlay_container = ft.Container(
            content=zoom_container,
            alignment=ft.alignment.center,
            expand=True,
            bgcolor=ft.Colors.BLACK54,
            padding=50
        )

        # اضافه کردن به overlay
        page.overlay.append(zoom_overlay_container)
        page.update()

    # نمایش اطلاعات تصویر
    def show_image_info(hit):
        tags = hit.get("tags", "").split(",")
        user = hit.get("user", "ناشناس")
        likes = hit.get("likes", 0)
        views = hit.get("views", 0)
        downloads = hit.get("downloads", 0)
        comments = hit.get("comments", 0)

        # ایجاد چیپ‌های برچسب
        tag_chips = []
        for tag in tags[:5]:
            tag_chips.append(
                ft.Chip(
                    label=ft.Text(tag.strip(), size=12),
                    bgcolor=ft.Colors.BLUE_100,
                )
            )

        info_content = ft.Column([
            ft.ListTile(
                leading=ft.Icon(ft.Icons.PERSON),
                title=ft.Text("عکاس"),
                subtitle=ft.Text(user, weight=ft.FontWeight.BOLD),
            ),
            ft.ListTile(
                leading=ft.Icon(ft.Icons.FAVORITE),
                title=ft.Text("تعداد لایک"),
                subtitle=ft.Text(str(likes)),
            ),
            ft.ListTile(
                leading=ft.Icon(ft.Icons.REMOVE_RED_EYE),
                title=ft.Text("تعداد بازدید"),
                subtitle=ft.Text(str(views)),
            ),
            ft.ListTile(
                leading=ft.Icon(ft.Icons.DOWNLOAD),
                title=ft.Text("تعداد دانلود"),
                subtitle=ft.Text(str(downloads)),
            ),
            ft.ListTile(
                leading=ft.Icon(ft.Icons.COMMENT),
                title=ft.Text("تعداد نظرات"),
                subtitle=ft.Text(str(comments)),
            ),
            ft.Divider(),
            ft.Text("برچسب‌ها:", size=14, weight=ft.FontWeight.BOLD),
            ft.Row(
                tag_chips,
                wrap=True,
                spacing=5,
                run_spacing=5
            )
        ], scroll=ft.ScrollMode.ADAPTIVE)

        def close_info_dialog(e):
            info_dialog.open = False
            page.update()

        info_dialog = ft.AlertDialog(
            title=ft.Text("اطلاعات تصویر"),
            content=info_content,
            actions=[
                ft.TextButton("بستن", on_click=close_info_dialog)
            ]
        )

        page.dialog = info_dialog
        info_dialog.open = True
        page.update()

    # دانلود تصویر
    def download_image(hit):
        nonlocal selected_hit
        selected_hit = hit

        # تنظیم نام پیش‌فرض برای تصویر
        default_name = hit.get("tags", "image").split(",")[0].strip()
        # حذف کاراکترهای غیرمجاز در نام فایل
        import re
        default_name = re.sub(r'[<>:"/\\|?*]', '', default_name)
        image_name_input.value = default_name

        # تنظیم کیفیت دانلود پیش‌فرض
        download_resolution.value = user_settings["download_quality"]

        # تنظیمات پیش‌فرض بهبود کیفیت
        enhancement_type_dropdown.value = user_settings["default_enhancement"]
        enhancement_level_dropdown.value = user_settings["default_enhancement_level"]
        upscale_dropdown.value = user_settings["default_upscale"]

        # نمایش دیالوگ دانلود
        download_dialog.open = True
        page.update()

    # ذخیره تصویر - نسخه پیشرفته
    def on_path_selected(e):
        if e.path and image_name_input.value and selected_hit:
            # اطمینان از فرمت فایل
            file_name = image_name_input.value.strip()
            if not file_name:
                show_snackbar("لطفاً نام فایل را وارد کنید")
                return

            if not file_name.lower().endswith(('.jpg', '.jpeg', '.png')):
                file_name += ".jpg"

            download_path = os.path.join(e.path, file_name)

            try:
                # ایجاد پوشه اگر وجود ندارد
                os.makedirs(os.path.dirname(download_path), exist_ok=True)

                # استفاده از کیفیت انتخابی برای دانلود
                download_url = selected_hit.get(download_resolution.value, selected_hit["largeImageURL"])

                # دانلود تصویر
                response = requests.get(download_url, timeout=30)
                response.raise_for_status()
                image_data = response.content

                # اعمال بهبود کیفیت اگر کاربر انتخاب کرده باشد
                enhancement_type = enhancement_type_dropdown.value
                enhancement_level = enhancement_level_dropdown.value
                upscale_factor = upscale_dropdown.value

                improvements_applied = []

                # اگر هرگونه بهبودی انتخاب شده
                if enhancement_type != "none" or upscale_factor != "none":
                    enhanced_data, original_size, new_size = enhance_image_local(
                        image_data, enhancement_type, enhancement_level, upscale_factor
                    )

                    if enhanced_data:
                        image_data = enhanced_data

                        # ثبت بهبودهای اعمال شده
                        if upscale_factor != "none":
                            improvements_applied.append(f"رزولوشن {upscale_factor}")

                        if enhancement_type != "none":
                            improvements_applied.append(f"{enhancement_type} ({enhancement_level})")

                        if improvements_applied:
                            show_snackbar(f"✅ بهبودها اعمال شد: {', '.join(improvements_applied)}")

                # ذخیره فایل
                with open(download_path, 'wb') as handler:
                    handler.write(image_data)

                show_snackbar(f"تصویر با موفقیت ذخیره شد: {file_name}")
                download_dialog.open = False
                page.update()

            except requests.exceptions.RequestException as e:
                show_snackbar(f"خطا در دانلود تصویر: {str(e)}")
            except Exception as e:
                show_snackbar(f"خطا در ذخیره فایل: {str(e)}")
        else:
            show_snackbar("مسیر یا نام تصویر نامعتبر است.")

    # نمایش تصاویر
    def show_images(query, page_num=1, append=False):
        nonlocal is_loading, selected_category, current_page_num
        if is_loading:
            return

        is_loading = True

        # اضافه کردن دسته‌بندی قبلی به تاریخچه فقط اگر از صفحه اصلی نباشد
        if selected_category and selected_category != query and gallery.visible:
            previous_categories.append(selected_category)

        selected_category = query
        current_page_num = page_num

        loading_indicator.visible = True
        load_more_button.visible = False
        page.update()

        hits = fetch_images(query, page_num)

        if append:
            gallery.controls.extend([create_image_control(hit) for hit in hits])
        else:
            gallery.controls = [create_image_control(hit) for hit in hits]

        if not gallery.controls and not append:
            gallery.controls = [
                ft.Container(
                    content=ft.Column([
                        ft.Icon(ft.Icons.SEARCH_OFF, size=64, color=ft.Colors.GREY_400),
                        ft.Text("هیچ تصویری یافت نشد.", size=16, color=ft.Colors.GREY_600)
                    ], alignment=ft.MainAxisAlignment.CENTER, horizontal_alignment=ft.CrossAxisAlignment.CENTER),
                    alignment=ft.alignment.center,
                    padding=20
                )
            ]

        loading_indicator.visible = False

        # نمایش دکمه بارگذاری بیشتر
        load_more_button.visible = gallery.visible and len(hits) >= user_settings["images_per_page"]

        # همچنین مقدار دسته‌بندی در منو را به روز کنیم
        for cat in categories:
            if cat["query"] == query:
                category_menu.value = query
                break

        # پاکسازی کش قدیمی
        clear_old_cache()

        page.update()
        is_loading = False

    # بارگذاری تصاویر بیشتر
    def load_more_images(e):
        nonlocal current_page_num
        if gallery.visible and not is_loading:
            current_page_num += 1
            show_images(selected_category, page_num=current_page_num, append=True)

    # بازگشت به صفحه قبل
    def go_back(e):
        if previous_categories:
            last_category = previous_categories.pop()
            show_images(last_category)
            # وضعیت المان‌ها برای حالت گالری
            home_container.visible = False
            gallery.visible = True
            back_button.visible = True
            category_menu.visible = True
            resolution_menu.visible = True
            load_more_button.visible = True
        elif gallery.visible:
            show_home()
        else:
            show_snackbar("دسته‌بندی قبلی وجود ندارد")
        page.update()

    # نمایش صفحه اصلی
    def show_home():
        nonlocal selected_category, current_page_num
        selected_category = None
        current_page_num = 1
        home_container.visible = True
        gallery.visible = False
        back_button.visible = False
        category_menu.visible = False
        resolution_menu.visible = False
        load_more_button.visible = False
        loading_indicator.visible = False
        page.update()

    # انتخاب دسته‌بندی از صفحه اصلی
    def select_category(e):
        # پاک کردن تاریخچه هنگام انتخاب از صفحه اصلی
        previous_categories.clear()
        show_images(e.control.data)
        home_container.visible = False
        gallery.visible = True
        back_button.visible = True
        category_menu.visible = True
        resolution_menu.visible = True
        load_more_button.visible = True
        page.update()

    # انتخاب دسته‌بندی از منو
    def select_category_from_menu(e):
        if e.control.value:
            show_images(e.control.value)
            home_container.visible = False
            gallery.visible = True
            back_button.visible = True
            category_menu.visible = True
            resolution_menu.visible = True
            load_more_button.visible = True
            page.update()

    # تغییر رزولوشن نمایش
    def set_resolution(resolution):
        nonlocal selected_resolution
        selected_resolution = resolution
        user_settings["image_quality"] = resolution
        save_settings(user_settings)
        if selected_category:
            show_images(selected_category)

    # تغییر تم
    def change_theme(theme_name):
        user_settings["primary_color"] = color_name_to_value[theme_name]
        save_settings(user_settings)
        apply_theme()

    # تغییر حالت تاریک/روشن
    def toggle_theme_mode(e=None):
        if page.theme_mode == ft.ThemeMode.LIGHT:
            page.theme_mode = ft.ThemeMode.DARK
            user_settings["theme_mode"] = "dark"
        else:
            page.theme_mode = ft.ThemeMode.LIGHT
            user_settings["theme_mode"] = "light"
        save_settings(user_settings)
        apply_theme()

    # اعمال تم
    def apply_theme():
        current_color = get_color_from_setting(user_settings["primary_color"])
        page.theme = ft.Theme(
            color_scheme_seed=current_color,
            use_material3=True,
        )
        page.update()

    # باز کردن تنظیمات
    def open_settings(e):
        # به روزرسانی مقادیر کنترل‌ها با تنظیمات فعلی
        theme_dropdown.value = color_value_to_name.get(user_settings["primary_color"], "آبی")
        image_quality_dropdown.value = user_settings["image_quality"]
        download_quality_dropdown.value = user_settings["download_quality"]
        images_per_page_dropdown.value = str(user_settings["images_per_page"])
        cache_switch.value = user_settings["enable_cache"]
        default_enhancement_dropdown.value = user_settings["default_enhancement"]
        default_enhancement_level_dropdown.value = user_settings["default_enhancement_level"]
        default_upscale_dropdown.value = user_settings["default_upscale"]

        settings_dialog.open = True
        page.update()

    # ذخیره تنظیمات
    def save_app_settings(e):
        user_settings["primary_color"] = color_name_to_value[theme_dropdown.value]
        user_settings["image_quality"] = image_quality_dropdown.value
        user_settings["download_quality"] = download_quality_dropdown.value
        user_settings["images_per_page"] = int(images_per_page_dropdown.value)
        user_settings["enable_cache"] = cache_switch.value
        user_settings["default_enhancement"] = default_enhancement_dropdown.value
        user_settings["default_enhancement_level"] = default_enhancement_level_dropdown.value
        user_settings["default_upscale"] = default_upscale_dropdown.value

        # پاکسازی کش اگر غیرفعال شده
        if not user_settings["enable_cache"]:
            image_cache.clear()

        save_settings(user_settings)
        apply_theme()
        settings_dialog.open = False
        page.update()
        show_snackbar("تنظیمات ذخیره شد")

    # پاکسازی کش
    def clear_cache(e):
        image_cache.clear()
        show_snackbar("کش تصاویر پاک شد")

    # ایجاد ویجت‌ها

    # File Picker برای انتخاب مسیر دانلود
    file_picker = ft.FilePicker(on_result=on_path_selected)
    page.overlay.append(file_picker)

    # صفحه اصلی
    home_container = ft.Container(
        content=ft.Column(
            [
                ft.Text("گالری تصاویر آنلاین",
                        size=32,
                        weight=ft.FontWeight.BOLD,
                        text_align=ft.TextAlign.CENTER),
                ft.Text("یک دسته‌بندی را انتخاب کنید",
                        size=18,
                        color=ft.Colors.GREY_700,
                        text_align=ft.TextAlign.CENTER),
                ft.GridView(
                    expand=True,
                    max_extent=180,
                    child_aspect_ratio=1.0,
                    padding=15,
                    spacing=10,
                    run_spacing=10,
                    controls=[
                        ft.Container(
                            content=ft.Column(
                                [
                                    ft.Icon(cat["icon"], size=35,
                                            color=get_color_from_setting(user_settings["primary_color"])),
                                    ft.Text(cat["name"], size=14, weight=ft.FontWeight.BOLD,
                                            text_align=ft.TextAlign.CENTER)
                                ],
                                alignment=ft.MainAxisAlignment.CENTER,
                                horizontal_alignment=ft.CrossAxisAlignment.CENTER,
                                spacing=8
                            ),
                            data=cat["query"],
                            on_click=select_category,
                            border_radius=12,
                            padding=15,
                            bgcolor=ft.Colors.SURFACE,
                            shadow=ft.BoxShadow(blur_radius=8, color=ft.Colors.BLACK12),
                            alignment=ft.alignment.center,
                        ) for cat in categories
                    ]
                )
            ],
            alignment=ft.MainAxisAlignment.START,
            horizontal_alignment=ft.CrossAxisAlignment.CENTER,
            spacing=20
        ),
        visible=True,
        expand=True,
        padding=15
    )

    # منوی دسته‌بندی
    category_menu = ft.Dropdown(
        options=[ft.dropdown.Option(text=cat["name"], key=cat["query"]) for cat in categories],
        on_change=select_category_from_menu,
        visible=False,
        label="دسته‌بندی",
        width=200
    )

    # منوی کیفیت نمایش
    resolution_menu = ft.Dropdown(
        options=[
            ft.dropdown.Option(text="کیفیت پایین", key="webformatURL"),
            ft.dropdown.Option(text="کیفیت بالا", key="largeImageURL")
        ],
        value=user_settings["image_quality"],
        on_change=lambda e: set_resolution(e.control.value),
        visible=False,
        label="کیفیت نمایش",
        width=150
    )

    # گالری تصاویر
    gallery = ft.GridView(
        expand=True,
        max_extent=230,
        child_aspect_ratio=1.2,
        padding=20,
        spacing=15,
        run_spacing=15,
        visible=False
    )

    # دکمه بارگذاری بیشتر
    load_more_button = ft.Container(
        content=ft.ElevatedButton(
            "بارگذاری تصاویر بیشتر",
            icon=ft.Icons.ADD,
            on_click=load_more_images,
            style=ft.ButtonStyle(
                shape=ft.RoundedRectangleBorder(radius=10),
                padding=20
            )
        ),
        alignment=ft.alignment.center,
        padding=20,
        visible=False
    )

    # نشانگر بارگذاری
    loading_indicator = ft.Container(
        content=ft.Column([
            ft.ProgressRing(),
            ft.Text("در حال بارگذاری...", size=14)
        ], alignment=ft.MainAxisAlignment.CENTER, horizontal_alignment=ft.CrossAxisAlignment.CENTER),
        alignment=ft.alignment.center,
        visible=False
    )

    # دکمه بازگشت
    back_button = ft.ElevatedButton(
        icon=ft.Icons.ARROW_BACK,
        text="بازگشت",
        on_click=go_back,
        visible=False,
        style=ft.ButtonStyle(
            shape=ft.RoundedRectangleBorder(radius=10),
            padding=15
        )
    )

    # دکمه تنظیمات
    settings_button = ft.IconButton(
        icon=ft.Icons.SETTINGS,
        tooltip="تنظیمات",
        on_click=open_settings
    )

    # دکمه تغییر حالت تاریک/روشن
    theme_toggle_button = ft.IconButton(
        icon=ft.Icons.BRIGHTNESS_4,
        tooltip="تغییر حالت تاریک/روشن",
        on_click=toggle_theme_mode
    )

    # AppBar
    appbar = ft.AppBar(
        title=ft.Text("گالری آنلاین", weight=ft.FontWeight.BOLD),
        center_title=True,
        bgcolor=ft.Colors.SURFACE,
        actions=[
            category_menu,
            resolution_menu,
            theme_toggle_button,
            settings_button
        ]
    )

    # دیالوگ دانلود - با قابلیت بهبود کیفیت پیشرفته
    image_name_input = ft.TextField(
        label="نام تصویر",
        hint_text="نام فایل تصویر را وارد کنید",
        expand=True
    )

    download_resolution = ft.Dropdown(
        label="کیفیت دانلود",
        options=[
            ft.dropdown.Option(text="کیفیت پایین", key="webformatURL"),
            ft.dropdown.Option(text="کیفیت بالا", key="largeImageURL"),
            ft.dropdown.Option(text="کیفیت فول اچ‌دی", key="fullHDURL"),
            ft.dropdown.Option(text="اصلی", key="imageURL"),
        ],
        width=200
    )

    # منوی بهبود کیفیت در صفحه دانلود
    enhancement_type_dropdown = ft.Dropdown(
        label="نوع بهبود",
        options=[
            ft.dropdown.Option(text="بدون بهبود", key="none"),
            ft.dropdown.Option(text="وضوح و شارپنس", key="sharpness"),
            ft.dropdown.Option(text="کنتراست و رنگ", key="contrast"),
            ft.dropdown.Option(text="روشنایی", key="brightness"),
            ft.dropdown.Option(text="همه بهبودها", key="all"),
        ],
        width=200,
        value="none"
    )

    enhancement_level_dropdown = ft.Dropdown(
        label="سطح بهبود",
        options=[
            ft.dropdown.Option(text="ملایم", key="low"),
            ft.dropdown.Option(text="متوسط", key="medium"),
            ft.dropdown.Option(text="قوی", key="high"),
            ft.dropdown.Option(text="خیلی قوی", key="ultra"),
        ],
        width=150,
        value="medium"
    )

    upscale_dropdown = ft.Dropdown(
        label="افزایش رزولوشن",
        options=[
            ft.dropdown.Option(text="بدون افزایش", key="none"),
            ft.dropdown.Option(text="۲ برابر (2x)", key="2x"),
            ft.dropdown.Option(text="۴ برابر (4x)", key="4x"),
        ],
        width=150,
        value="none"
    )

    def close_download_dialog(e):
        download_dialog.open = False
        page.update()

    download_dialog = ft.AlertDialog(
        title=ft.Text("ذخیره تصویر با کیفیت پیشرفته"),
        content=ft.Column([
            ft.Row([image_name_input, download_resolution]),
            ft.Divider(),
            ft.Text("🔄 تنظیمات بهبود کیفیت:", size=14, weight=ft.FontWeight.BOLD),
            ft.Row([enhancement_type_dropdown, enhancement_level_dropdown]),
            ft.Text("📐 افزایش رزولوشن:", size=14, weight=ft.FontWeight.BOLD),
            ft.Row([upscale_dropdown]),
            ft.Container(
                content=ft.Column([
                    ft.Text("💡 نکات:", size=12, weight=ft.FontWeight.BOLD),
                    ft.Text("• افزایش رزولوشن: سایز تصویر را بزرگ‌تر می‌کند", size=10),
                    ft.Text("• بهبود وضوح: جزئیات را واضح‌تر می‌کند", size=10),
                    ft.Text("• بهبود کنتراست: رنگ‌ها را زنده‌تر می‌کند", size=10),
                    ft.Text("⚠️ افزایش رزولوشن زمان بیشتری نیاز دارد", size=10, color=ft.Colors.ORANGE),
                ]),
                bgcolor=ft.Colors.BLUE_50,
                padding=10,
                border_radius=10
            ),
            ft.Text("لطفاً مسیر ذخیره‌سازی را انتخاب کنید", size=12, color=ft.Colors.GREY_600)
        ]),
        actions=[
            ft.TextButton(
                text="📁 انتخاب مسیر",
                icon=ft.Icons.FOLDER_OPEN,
                on_click=lambda _: file_picker.get_directory_path()
            ),
            ft.TextButton(
                text="🚫 لغو",
                on_click=close_download_dialog
            )
        ]
    )

    # سوئیچ فعال‌سازی کش
    cache_switch = ft.Switch(
        label="فعال‌سازی کش تصاویر",
        value=user_settings["enable_cache"]
    )

    # دیالوگ تنظیمات
    theme_dropdown = ft.Dropdown(
        label="رنگ تم",
        options=[ft.dropdown.Option(text=name) for name in color_themes.keys()],
        width=200
    )

    image_quality_dropdown = ft.Dropdown(
        label="کیفیت نمایش تصاویر",
        options=[
            ft.dropdown.Option(text="کیفیت پایین", key="webformatURL"),
            ft.dropdown.Option(text="کیفیت بالا", key="largeImageURL")
        ],
        width=200
    )

    download_quality_dropdown = ft.Dropdown(
        label="کیفیت پیش‌فرض دانلود",
        options=[
            ft.dropdown.Option(text="کیفیت پایین", key="webformatURL"),
            ft.dropdown.Option(text="کیفیت بالا", key="largeImageURL"),
            ft.dropdown.Option(text="کیفیت فول اچ‌دی", key="fullHDURL"),
            ft.dropdown.Option(text="اصلی", key="imageURL"),
        ],
        width=200
    )

    images_per_page_dropdown = ft.Dropdown(
        label="تعداد تصاویر در هر صفحه",
        options=[ft.dropdown.Option(text=str(i)) for i in [12, 24, 36, 48, 60]],
        width=200
    )

    # تنظیمات پیش‌فرض بهبود کیفیت
    default_enhancement_dropdown = ft.Dropdown(
        label="بهبود کیفیت پیش‌فرض",
        options=[
            ft.dropdown.Option(text="بدون بهبود", key="none"),
            ft.dropdown.Option(text="وضوح و شارپنس", key="sharpness"),
            ft.dropdown.Option(text="کنتراست و رنگ", key="contrast"),
            ft.dropdown.Option(text="روشنایی", key="brightness"),
            ft.dropdown.Option(text="همه بهبودها", key="all"),
        ],
        width=200
    )

    default_enhancement_level_dropdown = ft.Dropdown(
        label="سطح بهبود پیش‌فرض",
        options=[
            ft.dropdown.Option(text="ملایم", key="low"),
            ft.dropdown.Option(text="متوسط", key="medium"),
            ft.dropdown.Option(text="قوی", key="high"),
            ft.dropdown.Option(text="خیلی قوی", key="ultra"),
        ],
        width=200
    )

    default_upscale_dropdown = ft.Dropdown(
        label="افزایش رزولوشن پیش‌فرض",
        options=[
            ft.dropdown.Option(text="بدون افزایش", key="none"),
            ft.dropdown.Option(text="۲ برابر (2x)", key="2x"),
            ft.dropdown.Option(text="۴ برابر (4x)", key="4x"),
        ],
        width=200
    )

    settings_dialog = ft.AlertDialog(
        title=ft.Text("تنظیمات برنامه"),
        content=ft.Column([
            theme_dropdown,
            image_quality_dropdown,
            download_quality_dropdown,
            images_per_page_dropdown,
            ft.Row([
                cache_switch,
                ft.ElevatedButton(
                    text="پاکسازی کش",
                    icon=ft.Icons.CLEANING_SERVICES,
                    on_click=clear_cache
                )
            ]),
            ft.Divider(),
            ft.Text("تنظیمات پیش‌فرض بهبود کیفیت:", size=14, weight=ft.FontWeight.BOLD),
            ft.Row([default_enhancement_dropdown, default_enhancement_level_dropdown]),
            ft.Row([default_upscale_dropdown]),
        ], scroll=ft.ScrollMode.ADAPTIVE),
        actions=[
            ft.TextButton("ذخیره", on_click=save_app_settings),
            ft.TextButton("لغو", on_click=lambda e: setattr(settings_dialog, 'open', False) or page.update())
        ]
    )

    # اضافه کردن المان‌ها به صفحه
    page.appbar = appbar

    page.add(
        ft.Column([
            ft.Row([back_button], alignment=ft.MainAxisAlignment.START),
            home_container,
            gallery,
            loading_indicator,
            load_more_button
        ], expand=True)
    )

    # اضافه کردن دیالوگ‌ها به overlay
    page.overlay.extend([download_dialog, settings_dialog])

    # اعمال تم اولیه
    apply_theme()


# اجرای برنامه
if __name__ == "__main__":
    ft.app(target=main)